import logging

from google.protobuf import json_format
from library.python.protobuf.json import (
    json2proto,
    proto2json,
)
import requests
import retry

import crypta.lib.python.tvm.helpers as tvm
from crypta.lib.python import http_client_utils
from crypta.siberia.bin.core.proto.add_user_set_response_pb2 import TAddUserSetResponse
from crypta.siberia.bin.core.proto.users_response_pb2 import TUsersResponse
from crypta.siberia.bin.common.data.proto import segment_pb2
from crypta.siberia.bin.common.data.proto.user_set_pb2 import TUserSet
from crypta.siberia.bin.common.proto.describe_ids_response_pb2 import TDescribeIdsResponse
from crypta.siberia.bin.common.proto.stats_pb2 import TStats


logger = logging.getLogger(__name__)


def with_logging(f):
    def wrapper(*args, **kwargs):
        response = f(*args, **kwargs)

        logger.debug("status code: %s", response.status_code)
        try:
            logger.debug("response: %s", response.json())
        except Exception:
            logger.debug("response: %s", response.text)

        return response

    return wrapper


def with_users_response(f):
    return http_client_utils.with_protobuf_response(TUsersResponse, json2proto.Json2ProtoConfig(map_as_object=True))(f)


def with_stats(f):
    return http_client_utils.with_protobuf_response(TStats)(f)


class SiberiaClient(object):
    def __init__(self, host, port):
        self.host = host
        self.port = port
        self.url = "http://{host}:{port}/{path}?@client=crypta-siberia-python-client"
        self.session = requests.Session()
        adapter = requests.adapters.HTTPAdapter(pool_connections=25, pool_maxsize=50)
        self.session.mount('http://', adapter)

    def get_url(self, path):
        return self.url.format(host=self.host, port=self.port, path=path)

    @retry.retry(exceptions=requests.ConnectionError, tries=2, delay=0.1, logger=logger)
    @with_logging
    def _get(self, path, params, tvm_ticket=None):
        url = self.get_url(path)
        params = {k: v for k, v in params.items() if v is not None}
        logger.debug("url: %s", url)
        logger.debug("params: %s", params)
        return self.session.get(url, params=params, headers=tvm.get_tvm_headers(tvm_ticket))

    @retry.retry(exceptions=requests.ConnectionError, tries=2, delay=0.1, logger=logger)
    @with_logging
    def _post(self, path, params, data=None, tvm_ticket=None):
        url = self.get_url(path)
        params = {k: v for k, v in params.items() if v is not None}
        logger.debug("url: %s", url)
        logger.debug("params: %s", params)
        logger.debug("data: %s", data)
        return self.session.post(url, params=params, data=data, headers=tvm.get_tvm_headers(tvm_ticket))

    @retry.retry(exceptions=requests.ConnectionError, tries=2, delay=0.1, logger=logger)
    @with_logging
    def _delete(self, path, params, data, tvm_ticket=None):
        url = self.get_url(path)
        params = {k: v for k, v in params.items() if v is not None}
        logger.debug("url: %s", url)
        logger.debug("params: %s", params)
        logger.debug("data: %s", data)
        return self.session.delete(url, params=params, data=data, headers=tvm.get_tvm_headers(tvm_ticket))

    @http_client_utils.with_simple_response
    def ping(self, tvm_ticket=None):
        return self._get("ping", {}, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def version(self, tvm_ticket=None):
        return self._get("version", {}, tvm_ticket=tvm_ticket)

    @http_client_utils.with_protobuf_response(TAddUserSetResponse)
    def user_sets_add(self, title, ttl=None, tvm_ticket=None):
        params = dict(title=title, ttl=ttl)
        return self._post("user_sets/add", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def user_sets_update(self, user_set_id, title=None, status=None, ttl=None, tvm_ticket=None):
        params = dict(user_set_id=user_set_id, title=title, status=status, ttl=ttl)
        return self._post("user_sets/update", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def user_sets_remove(self, user_set_id, tvm_ticket=None):
        params = dict(user_set_id=user_set_id)
        return self._delete("user_sets/remove", params, data=None, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def user_sets_remove_data(self, user_set_id, tvm_ticket=None):
        params = dict(user_set_id=user_set_id)
        return self._delete("user_sets/remove_data", params, data=None, tvm_ticket=tvm_ticket)

    @http_client_utils.with_protobuf_response(TUserSet)
    def user_sets_get(self, user_set_id, tvm_ticket=None):
        params = dict(user_set_id=user_set_id)
        return self._get("user_sets/get", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def user_sets_describe(self, user_set_id, tvm_ticket=None):
        params = dict(user_set_id=user_set_id)
        return self._post("user_sets/describe", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_protobuf_response(TDescribeIdsResponse)
    def user_sets_describe_ids(self, ids, mode=None, experiment=None, ttl=None, tvm_ticket=None):
        if experiment is not None:
            experiment = proto2json.proto2json(experiment)

        params = dict(mode=mode, experiment=experiment, ttl=ttl)
        return self._post("user_sets/describe_ids", params, json_format.MessageToJson(ids, indent=None), tvm_ticket=tvm_ticket)

    @with_stats
    def user_sets_get_stats(self, user_set_id, tvm_ticket=None):
        params = dict(user_set_id=user_set_id)
        return self._get("user_sets/get_stats", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def users_add(self, user_set_id, users, tvm_ticket=None):
        params = dict(user_set_id=user_set_id)
        return self._post("users/add", params, data=proto2json.proto2json(users, config=proto2json.Proto2JsonConfig(map_as_object=True)), tvm_ticket=tvm_ticket)

    @with_users_response
    def users_search(self, user_set_id, limit=None, last_user_id=None, tvm_ticket=None):
        params = dict(user_set_id=user_set_id, limit=limit, last_user_id=last_user_id)
        return self._get("users/search", params, tvm_ticket=tvm_ticket)

    @with_stats
    def users_get_stats(self, yandexuid, tvm_ticket=None):
        params = dict(yandexuid=yandexuid)
        return self._get("users/get_stats", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_protobuf_response(segment_pb2.TSegment)
    def segments_make(self, user_set_id, title, rule, tvm_ticket=None):
        params = dict(user_set_id=user_set_id, title=title, rule=rule)
        return self._post("segments/make", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_protobuf_response(segment_pb2.TSegments)
    def segments_search(self, user_set_id, limit=None, last_segment_id=None, tvm_ticket=None):
        params = dict(user_set_id=user_set_id, limit=limit, last_segment_id=last_segment_id)
        return self._get("segments/search", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def segments_remove(self, user_set_id, segment_ids, tvm_ticket=None):
        params = dict(user_set_id=user_set_id)
        data = ",".join(str(id) for id in segment_ids)
        return self._post("segments/remove", params, data, tvm_ticket=tvm_ticket)

    @with_users_response
    def segments_list_users(self, user_set_id, segment_id, limit=None, last_user_id=None, tvm_ticket=None):
        params = dict(user_set_id=user_set_id, segment_id=segment_id, limit=limit, last_user_id=last_user_id)
        return self._get("segments/list_users", params, tvm_ticket=tvm_ticket)

    @http_client_utils.with_simple_response
    def segments_describe(self, user_set_id, segment_id, tvm_ticket=None):
        params = dict(user_set_id=user_set_id, segment_id=segment_id)
        return self._post("segments/describe", params, tvm_ticket=tvm_ticket)

    @with_stats
    def segments_get_stats(self, user_set_id, segment_id, tvm_ticket=None):
        params = dict(user_set_id=user_set_id, segment_id=segment_id)
        return self._get("segments/get_stats", params, tvm_ticket=tvm_ticket)
