#include "siberia_client.h"

#include <crypta/lib/native/tvm/headers.h>
#include <crypta/siberia/bin/common/proto/describe_ids_response.pb.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/string/builder.h>

using namespace NCrypta;
using namespace NCrypta::NSiberia;

TSiberiaClient::TSiberiaClient(const TString& host, ui32 port, TDuration timeout, const NTvmAuth::TTvmClient& tvmClient, TRetryOptions retryOptions)
    : Address(TStringBuilder() << host << ":" << port)
    , TvmClient(tvmClient)
    , RetryOptions(std::move(retryOptions))
    , Timeout(std::move(timeout))
{}

TUserSetId TSiberiaClient::DescribeIds(const TIds& ids, const TMaybe<TString> mode, const TMaybe<TDescribingExperiment> experiment) const {
    TCgiParameters cgiParameters;

    if (mode.Defined()){
        cgiParameters.InsertUnescaped(DESCRIBING_MODE_CGI_PARAM, *mode);
    }
    if (experiment.Defined()){
        cgiParameters.InsertUnescaped(EXPERIMENT_CGI_PARAM, NProtobufJson::Proto2Json(*experiment));
    }

    const auto& url = GetUrl("post", "/user_sets/describe_ids", cgiParameters);
    const auto& message = BuildMessage(url, NProtobufJson::Proto2Json(ids), "application/json");
    const auto& response = SendMessageWithRetries(message);
    return FromString<TUserSetId>(NProtobufJson::Json2Proto<TDescribeIdsResponse>(response->Data).GetUserSetId());
}

TStats TSiberiaClient::GetUserSetStats(TUserSetId userSetId) const {
    const auto& url = GetUrl("http", "/user_sets/get_stats", {{"user_set_id", ToString(userSetId)}});
    const auto& message = BuildMessage(url, "");
    const auto& response = SendMessageWithRetries(message);
    return NProtobufJson::Json2Proto<TStats>(response->Data);
}

TString TSiberiaClient::GetUrl(const TString& scheme, const TString& path, const TCgiParameters& cgiParameters) const {
    return TStringBuilder() << scheme <<  "://" << Address << path << "?" << cgiParameters.Print();
}

NNeh::TResponseRef TSiberiaClient::SendMessageWithRetries(const NNeh::TMessage& message) const {
    try {
        NNeh::TResponseRef response;

        auto describeFunc = [&response, &message, this](){
            response = NNeh::Request(message)->Wait(Timeout);
            Y_ENSURE(response, "Timeout");
            Y_ENSURE(!response->IsError(), response->GetErrorText());
        };

        DoWithRetry<std::exception>(describeFunc, RetryOptions, true);

        return response;
    } catch (const std::exception& e) {
        ythrow TSiberiaClientException() << e.what();
    }
}

NNeh::TMessage TSiberiaClient::BuildMessage(const TString& url, const TStringBuf& content, const TStringBuf& contentType) const {
    NNeh::TMessage message(url, "");
    const bool isMessageBuilt = NNeh::NHttp::MakeFullRequest(
        message,
        GetSerializedTvmHeaders(TvmClient.GetServiceTicketFor(SIBERIA_TVM_ALIAS)),
        content,
        contentType
    );
    Y_ENSURE(isMessageBuilt, "Failed to build message");
    return message;
}
