#include "parse_utils.h"

#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/siberia/bin/common/ydb/tables/user_data_stats_table_fields.h>
#include <crypta/siberia/bin/common/ydb/tables/user_set_stats_fields.h>

#include <util/string/cast.h>

using namespace NCrypta;
using namespace NCrypta::NSiberia;

void NYdbCommonParseUtils::ParseUserDataStats(NLab::TUserDataStats& userDataStats, NYdb::TResultSetParser& parser, const TString& columnName) {
    NProtoSerializer::FromString(userDataStats, *parser.ColumnParser(columnName).GetOptionalString());
}

NLab::TUserDataStats NYdbCommonParseUtils::ParseUserDataStats(NYdb::TResultSetParser& parser, const TString& columnName) {
    NLab::TUserDataStats res;
    ParseUserDataStats(res, parser, columnName);
    return res;
}

TVector<NLab::TUserDataStats> NYdbCommonParseUtils::ParseUserDataStats(const NYdb::TResultSet& resultSet) {
    NYdb::TResultSetParser parser(resultSet);
    TVector<NLab::TUserDataStats> res;

    while (parser.TryNextRow()) {
        res.push_back(NYdbCommonParseUtils::ParseUserDataStats(parser, NUserDataStatsTableFields::STATS));
    }

    return res;
}

TMaybe<NLab::TUserDataStats> NYdbCommonParseUtils::ParseSingleUserDataStats(const NYdb::TResultSet& resultSet) {
    const auto& res = ParseUserDataStats(resultSet);
    Y_ENSURE(res.size() <= 1, "Too many records found");
    return !res.empty() ? TMaybe<NLab::TUserDataStats>(res.at(0)) : Nothing();
}


TMaybe<TUserSetStats> NYdbCommonParseUtils::ParseUserSetStats(const NYdb::TResultSet& resultSet) {
    NYdb::TResultSetParser parser(resultSet);

    if (parser.TryNextRow()) {
        return TUserSetStats {
            .ProcessedUsersCount = *parser.ColumnParser(NUserSetStatsTableFields::PROCESSED_USERS_COUNT).GetOptionalUint64(),
            .UserDataStats = NYdbCommonParseUtils::ParseUserDataStats(parser, NUserSetStatsTableFields::STATS),
            .Ready = *parser.ColumnParser(NUserSetStatsTableFields::READY).GetOptionalBool()
        };
    }
    return Nothing();
}

NYdbCommonParseUtils::TUserAttributesResultSetParser::TUserAttributesResultSetParser(const NYdb::TResultSet& resultSet)
    : ResultSetParser(resultSet)
{}

bool NYdbCommonParseUtils::TUserAttributesResultSetParser::TryNextRow() {
    return ResultSetParser.TryNextRow();
}

NYdbCommonParseUtils::TUserAttributesResultSetParser::TRow NYdbCommonParseUtils::TUserAttributesResultSetParser::GetRow() {
    return {
        .UserId = *ResultSetParser.ColumnParser("user_id").GetOptionalUint64(),
        .Key = *ResultSetParser.ColumnParser("attribute_key").GetOptionalUtf8(),
        .Value = *ResultSetParser.ColumnParser("attribute_value").GetOptionalUtf8()
    };
}

TUserSet NYdbCommonParseUtils::ParseUserSet(NYdb::TResultSetParser& parser) {
    TUserSet userSet;
    userSet.SetId(ToString(*parser.ColumnParser("id").GetOptionalUint64()));
    userSet.SetTitle(*parser.ColumnParser("title").GetOptionalUtf8());
    userSet.SetExpirationTime(*parser.ColumnParser("expiration_time").GetOptionalUint64());
    userSet.SetType(*parser.ColumnParser("type").GetOptionalUtf8());
    userSet.SetStatus(*parser.ColumnParser("status").GetOptionalUtf8());
    return userSet;
}

TMaybe<TUserSet> NYdbCommonParseUtils::ParseUserSet(const NYdb::TResultSet& resultSet) {
    NYdb::TResultSetParser parser(resultSet);

    if (!parser.TryNextRow()) {
        return Nothing();
    }

    const auto& userSet = NYdbCommonParseUtils::ParseUserSet(parser);

    Y_ENSURE(!parser.TryNextRow(), "Too many results");

    return userSet;
}

TSegment NYdbCommonParseUtils::ParseSegment(NYdb::TResultSetParser& parser) {
    TSegment segment;
    segment.SetId(ToString(*parser.ColumnParser("id").GetOptionalUint64()));
    segment.SetTitle(*parser.ColumnParser("title").GetOptionalUtf8());
    segment.SetRule(*parser.ColumnParser("rule").GetOptionalUtf8());
    segment.SetStatus(*parser.ColumnParser("status").GetOptionalUtf8());
    segment.SetSize(*parser.ColumnParser("size").GetOptionalUint64());
    segment.SetCreationTs(*parser.ColumnParser("creation_ts").GetOptionalUint64());
    return segment;
}
