#pragma once

#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/siberia/bin/common/data/proto/segment.pb.h>
#include <crypta/siberia/bin/common/data/types.h>
#include <crypta/siberia/bin/common/data/proto/user.pb.h>
#include <crypta/siberia/bin/common/ydb/paths/paths.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <util/string/subst.h>

namespace NCrypta::NSiberia {
    struct TAddUsersToSegmentDbRequest {
        struct TRequestParams {
            const TVector<TUserId>& UserIds;
            TSegmentId SegmentId = 0;
            ui64 Ts = 0;
        };

        static constexpr const char* const Query = R"(
            PRAGMA TablePathPrefix("%s");

            DECLARE $segmentId AS UInt64;
            DECLARE $ts AS UInt64;
            DECLARE $userIds AS 'List<Struct<
                id: Uint64>>';

            REPLACE INTO {user_segments_table}
            SELECT
                id AS user_id,
                $segmentId AS segment_id,
                $ts AS ts
            FROM AS_TABLE($userIds);

            REPLACE INTO {segment_users_table}
            SELECT
                id AS user_id,
                $segmentId AS segment_id
            FROM AS_TABLE($userIds);

            $current_segment_size = (
                SELECT
                    size
                FROM {segments_table}
                WHERE id == $segmentId
            );

            UPSERT INTO {segments_table} (id, size)
            VALUES ($segmentId, $current_segment_size + ListLength($userIds))
        )";

        static TString GetQuery(const TRequestParams&) {
            TString query = Query;
            SubstGlobal(query, "{user_segments_table}", YDB_PATHS.GetUserSegmentsTable());
            SubstGlobal(query, "{segment_users_table}", YDB_PATHS.GetSegmentUsersTable());
            SubstGlobal(query, "{segments_table}", YDB_PATHS.GetSegmentsTable());
            return query;
        }

        static NYdb::TParams GetParams(NYdb::TParamsBuilder&& paramsBuilder, const TRequestParams& params) {
            paramsBuilder.AddParam("$segmentId").Uint64(params.SegmentId).Build();
            paramsBuilder.AddParam("$ts").Uint64(params.Ts).Build();

            auto& userIdsParam = paramsBuilder.AddParam("$userIds");

            userIdsParam.BeginList();

            for (auto& userId : params.UserIds) {
                userIdsParam.AddListItem()
                    .BeginStruct()
                    .AddMember("id").Uint64(userId)
                    .EndStruct();
            }

            userIdsParam.EndList().Build();

            return paramsBuilder.Build();
        }
    };

    NYdb::NTable::TDataQueryResult ExecuteAddUsersToSegmentDbRequest(NYdb::NTable::TSession& session, const TString& path, TSegmentId segmentId, const TVector<TUserId>& userIds, ui64 ts, NYdb::NTable::TTxControl&& txControl);
    void AddUsersToSegment(TYdbClient& ydbClient, TUserSetId userSetId, TSegmentId segmentId, const TVector<TUserId>& userIds);
}
