#pragma once

#include "helpers.h"

#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/siberia/bin/common/data/types.h>
#include <crypta/siberia/bin/common/data/user_set_update_info.h>
#include <crypta/siberia/bin/common/ydb/paths/paths.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <market/library/maybe_monad/maybe_monad.h>

#include <util/string/subst.h>
#include <util/string/vector.h>

namespace NCrypta::NSiberia {
    struct TUpdateUserSetDbRequest {
        struct TRequestParams {
            TUserSetId UserSetId = 0;
            const TUserSetUpdateInfo& UserSetUpdateInfo;
        };

        static constexpr const char* const Query = R"(
            PRAGMA TablePathPrefix("%s");

            DECLARE $data AS "Struct<
                id: Uint64,
                title: Optional<Utf8>,
                status: Optional<Utf8>,
                expiration_time: Optional<Uint64>
            >";

            UPSERT INTO {user_sets_table}
            SELECT
                id, {fields_to_select}
            FROM AS_TABLE(AsList($data));
        )";

        static TString GetQuery(const TRequestParams& params) {
            TVector<TString> fieldsToSelect;

            AddFieldIfDefined(fieldsToSelect, params.UserSetUpdateInfo.Status, "status");
            AddFieldIfDefined(fieldsToSelect, params.UserSetUpdateInfo.Title, "title");
            AddFieldIfDefined(fieldsToSelect, params.UserSetUpdateInfo.ExpirationTime, "expiration_time");

            TString query = Query;
            SubstGlobal(query, "{user_sets_table}", YDB_PATHS.GetUserSetsTable());
            SubstGlobal(query, "{fields_to_select}", JoinVectorIntoString(fieldsToSelect, ", "));
            return query;
        }

        static NYdb::TParams GetParams(NYdb::TParamsBuilder&& paramsBuilder, const TRequestParams& params) {
            return paramsBuilder
                .AddParam("$data")
                .BeginStruct()
                .AddMember("id").Uint64(params.UserSetId)
                .AddMember("title").OptionalUtf8(params.UserSetUpdateInfo.Title)
                .AddMember("status").OptionalUtf8(params.UserSetUpdateInfo.Status)
                .AddMember("expiration_time").OptionalUint64(Map(params.UserSetUpdateInfo.ExpirationTime, [](TInstant instant){ return instant.Seconds(); }))
                .EndStruct().Build()
                .Build();
        }
    };

    NYdb::NTable::TAsyncDataQueryResult ExecuteUpdateUserSetDbRequest(TYdbClient& ydbClient, TUserSetId userSetId, const TUserSetUpdateInfo& userSetUpdateInfo);
    void UpdateUserSet(TYdbClient& ydbClient, TUserSetId userSetId, const TUserSetUpdateInfo& UserSetUpdateInfo);
}

