#include <crypta/lib/native/dns/caching_reverse_dns_resolver.h>
#include <crypta/lib/native/http/request_reply.h>
#include <crypta/lib/native/http/server.h>
#include <crypta/lib/native/juggler/juggler_client.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/log/wrappers/pqlib/pq_spdlog_wrapper.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/producer.h>
#include <crypta/lib/native/proto_secrets/remove_secrets.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/graphite/graphite_stats_sink.h>
#include <crypta/lib/native/stats/log/log_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/stats/utils.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/tvm/create_tvm_client.h>
#include <crypta/lib/native/tvm/tvm_status_reporter.h>
#include <crypta/lib/native/yaml/parse_yaml_file.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/siberia/bin/core/lib/command_senders/describing_sender.h>
#include <crypta/siberia/bin/core/lib/command_senders/mutation_sender.h>
#include <crypta/siberia/bin/core/lib/command_senders/segmentation_sender.h>
#include <crypta/siberia/bin/core/lib/configs/proto/clients/clients_config.pb.h>
#include <crypta/siberia/bin/core/lib/configs/proto/config.pb.h>
#include <crypta/siberia/bin/core/lib/logic/request_processor_factory.h>
#include <crypta/siberia/bin/core/lib/logic/worker.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/yaml/as/tstring.h>

#include <yt/yt/core/misc/shutdown.h>

#include <util/system/env.h>

using namespace NCrypta;
using namespace NCrypta::NPQ;
using namespace NCrypta::NSiberia;

namespace {
    struct TAllConfigs {
        TConfig AppConfig;
        TClientsConfig ClientsConfig;
    };

    TAllConfigs GetConfigsFromCmdArgs(int argc, char** argv) {
        TString appConfigFile;
        TString clientsConfigFile;

        auto opts = NLastGetopt::TOpts::Default();
        opts.AddHelpOption();
        opts.AddLongOption("config").StoreResult(&appConfigFile).RequiredArgument("CONFIG").Required();
        opts.AddLongOption("clients").StoreResult(&clientsConfigFile).RequiredArgument("CLIENTS").Required();
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);

        return {
            .AppConfig = Yaml2Proto<TConfig>(ParseYamlFile(appConfigFile)),
            .ClientsConfig = Yaml2Proto<TClientsConfig>(ParseYamlFile(clientsConfigFile))
        };
    }
}

int main(int argc, char** argv) {
    TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

    auto stderrLog = NLog::GetLog();

    try {
        TShiftedClock::FreezeTimestampFromEnv();

        const auto& allConfigs = GetConfigsFromCmdArgs(argc, argv);
        const auto& appConfig = allConfigs.AppConfig;

        NCrypta::NLog::RegisterLogs(appConfig.GetLogs());
        auto log = NLog::GetLog("main");

        log->info("Config:\n{}", NProtoSecrets::GetCopyWithoutSecrets(appConfig).DebugString());

        TYdbClient ydbClient(appConfig.GetYdb());

        const auto& statsSettings = GetStatsSettings(appConfig.GetStats());

        const auto tvmClient = std::make_shared<NTvmAuth::TTvmClient>(CreateTvmClient(appConfig.GetTvm()));

        TJugglerClient jugglerClient(appConfig.GetJuggler());
        TTvmStatusReporter tvmStatusReporter(*tvmClient, jugglerClient);
        tvmStatusReporter.Start();

        TVector<THolder<IStatsSink>> sinks;
        if (appConfig.HasGraphite()) {
            sinks.emplace_back(MakeHolder<TGraphiteStatsSink>(appConfig.GetGraphite()));
        }
        if (appConfig.HasSolomon()) {
            sinks.emplace_back(MakeHolder<TSolomonStatsSink>(appConfig.GetSolomon()));
        }
        sinks.emplace_back(MakeHolder<TLogStatsSink>("graphite"));

        auto statsWriter = GetStatsWriter(std::move(sinks));

        const auto& lbConfig = appConfig.GetLogbroker();
        auto pqLib = NPQLib::Create(lbConfig.GetPqLib());
        auto pqLibLogger = MakeIntrusive<NCrypta::NLog::TPqSpdlogWrapper>(lbConfig.GetPqlibLogName());
        auto credentialsProvider = lbConfig.GetUseTvm() ? NPersQueue::CreateTVMCredentialsProvider(tvmClient, pqLibLogger) : NPersQueue::CreateInsecureCredentialsProvider();

        TProducer accessLogProducer(pqLib, lbConfig.GetAccessLogProducer(), credentialsProvider, statsSettings);
        TProducer changeLogProducer(pqLib, lbConfig.GetChangeLogProducer(), credentialsProvider, statsSettings);
        TProducer describeLogProducer(pqLib, lbConfig.GetDescribeLogProducer(), credentialsProvider, statsSettings);
        TProducer describeSlowLogProducer(pqLib, lbConfig.GetDescribeSlowLogProducer(), credentialsProvider, statsSettings);
        TProducer segmentateLogProducer(pqLib, lbConfig.GetSegmentateLogProducer(), credentialsProvider, statsSettings);
        log->info("LB producers successfully started");

        TMutationSender mutationSender(changeLogProducer);
        TDescribingSender describingSender(describeLogProducer);
        TDescribingSender slowDescribingSender(describeSlowLogProducer);
        TSegmentationSender segmentationSender(segmentateLogProducer);

        TRequestProcessorFactory requestProcessorFactory(
            ydbClient,
            mutationSender,
            describingSender,
            slowDescribingSender,
            segmentationSender,
            appConfig.GetProcessors(),
            *tvmClient,
            statsSettings
        );

        TWorker worker(requestProcessorFactory, allConfigs.ClientsConfig, statsSettings);

        auto serverProcessFunc = [&worker](NHttp::TRequestReply& reply) {
            worker.ProcessRequest(reply);
        };

        TCachingReverseDnsResolver reverseDnsResolver;

        log->info("Start HTTP server");

        NHttp::TServer server(
            serverProcessFunc,
            appConfig.GetHttp(),
            *tvmClient,
            reverseDnsResolver,
            NewJsonLogEntryBuilderFactory(),
            accessLogProducer,
            statsSettings);

        stopSignalWaiter.Wait();
    } catch (const std::exception& e) {
        stderrLog->error("Exception in main(): {}", e.what());
    } catch (...) {
        stderrLog->error("Unknown exception in main()");
    }

    NYT::Shutdown();
}
