#include "add_users_request_processor.h"

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/siberia/bin/common/data/proto/user.pb.h>
#include <crypta/lib/native/http/format.h>
#include <crypta/siberia/bin/core/lib/ydb/requests/insert_users_db_request.h>

#include <util/string/cast.h>
#include <util/digest/city.h>

using namespace NCrypta::NSiberia;

namespace {
    TString CalcHash(const TUser::TInfo& userInfo) {
        TString attributesStr;

        const auto& attributes = userInfo.GetAttributes();
        TVector<std::pair<TString, TUser::TInfo::TAttributeValues>> sortedAttributes(attributes.size());
        PartialSortCopy(attributes.begin(), attributes.end(), sortedAttributes.begin(), sortedAttributes.end(), [](const auto& lhs, const auto& rhs) {
            return lhs.first < rhs.first;
        });

        for (const auto& [key, attributeValues] : sortedAttributes) {
            const auto& values = attributeValues.GetValues();
            TVector<TString> sortedValues(values.size());
            PartialSortCopy(values.begin(), values.end(), sortedValues.begin(), sortedValues.end());

            attributesStr += key;
            for (const auto& value : sortedValues) {
                attributesStr += value;
            }
        }
        return ToString(CityHash64(attributesStr));
    }

    TVector<TUser> GetUsers(const TAddUsersRequest& request) {
        TVector<TUser> users;

        for (const auto& userInfo : request.GetUsers()) {
            TUser user;
            user.MutableInfo()->CopyFrom(userInfo);
            user.SetId(CalcHash(userInfo));
            users.push_back(std::move(user));
        }

        SortBy(users, [](const TUser& user) {
            return user.GetId();
        });

        return users;
    }
}

TAddUsersProcessor::TAddUsersProcessor(TYdbClient& ydbClient, const TStats::TSettings& statsSettings)
    : TBase(NLog::GetLog("add_users"), TaggedSingleton<TStats, decltype(*this)>("processors.add_users", statsSettings), ydbClient)
{}

void TAddUsersProcessor::DoProcess(NHttp::TRequestReply& reply, const TInsertUsersRequest& request) {
    const auto& users = GetUsers(request.AddUsersRequest);
    InsertUsers(YdbClient, request.UserSetId, GetUsers(request.AddUsersRequest));

    SendResponse(reply, HTTP_OK, NHttp::GetSimpleResponse(TStringBuilder() << users.size() << " users added"));
}
