#pragma once

#include <crypta/lib/native/http/format.h>
#include <crypta/lib/native/http/request_reply.h>
#include <crypta/lib/native/time/utils.h>
#include <crypta/lib/native/ydb/ydb_error_exception.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/http/misc/parsed_request.h>

namespace NCrypta::NSiberia {
    class IRequestProcessor {
    public:
        virtual void Process(NHttp::TRequestReply& reply, const TString& clientName) = 0;
        virtual ~IRequestProcessor() = default;
    };

    template<typename TRequest, typename TRequestParser>
    class TRequestProcessor : public IRequestProcessor {
    public:
        explicit TRequestProcessor(NLog::TLogPtr log, TStats& stats)
            : Log(log)
            , Stats(stats)
        {}

        void Process(NHttp::TRequestReply& reply, const TString& clientName) override final {
            Y_ENSURE(!reply.IsReplied(), "TRequestReply must be ready for reply when passed to TRequestProcessor::Process");

            Stats.Count->Add("request.total.received");
            Stats.Count->Add("tvm_client." + clientName);

            TRequest request;
            try {
                request = TRequestParser::Parse(TCgiParameters(reply.GetRequestCgi()), reply.GetRequestBody());
            } catch (const yexception& e) {
                Stats.Count->Add("request.errors.parsing");
                SendResponse(reply, HTTP_BAD_REQUEST, NHttp::GetSimpleResponse(e.what()));
                return;
            }

            try {
                Process(reply, request);
            } catch (const TYdbErrorException& e) {
                SendInternalError(reply, TStringBuilder() << "(TYdbErrorException) " << e.what() << ". Issues = " << e.GetStatus().GetIssues().ToString());
            } catch (const std::exception& e) {
                SendInternalError(reply, TStringBuilder() << "(std::exception) " << e.what());
            } catch (...) {
                SendInternalError(reply, "Unknown exception");
            }
        }

    protected:
        virtual void Process(NHttp::TRequestReply& reply, const TRequest& request) = 0;

        void SendResponse(NHttp::TRequestReply& reply, HttpCodes httpCode, const TString& response) {
            reply.Reply(httpCode, response, Stats, Log);
        }

        void SendInternalError(NHttp::TRequestReply& reply, const TString& message) {
            reply.ReplyError(HTTP_INTERNAL_SERVER_ERROR, message, Stats, Log);
        }

        NLog::TLogPtr Log;
        TStats& Stats;
    };
}
