#include "describe_ids_request_processor.h"

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/http/format.h>
#include <crypta/lib/native/ydb/executer.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>
#include <crypta/siberia/bin/common/describing/experiment/cpp/invalid_describing_experiment_exception.h>
#include <crypta/siberia/bin/common/describing/experiment/cpp/validator.h>
#include <crypta/siberia/bin/common/describing/mode/cpp/describing_mode.h>
#include <crypta/siberia/bin/common/describing/ydb_requests/ydb_requests.h>
#include <crypta/siberia/bin/common/proto/describe_ids_response.pb.h>
#include <crypta/siberia/bin/core/lib/logic/common/helpers/ids.h>
#include <crypta/siberia/bin/core/lib/ydb/parse_utils/parse_utils.h>
#include <crypta/siberia/bin/core/lib/ydb/requests/add_user_set_db_request.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/datetime/base.h>
#include <util/string/builder.h>

using namespace NCrypta::NSiberia;
using namespace NYdb;
using namespace NYdb::NTable;

namespace {
    TString GetResponse(TUserSetId userSetId) {
        TDescribeIdsResponse response;
        response.SetUserSetId(ToString(userSetId));
        return NProtobufJson::Proto2Json(response);
    }
}

TDescribeIdsProcessor::TDescribeIdsProcessor(
    TYdbClient& ydbClient,
    TDescribingSender& describingSender,
    TDescribingSender& slowDescribingSender,
    const TStats::TSettings& statsSettings,
    size_t maxIdsCount,
    ui64 defaultTtlSeconds
)
    : TBase(NLog::GetLog("describe_ids"), TaggedSingleton<TStats, decltype(*this)>("processors.describe_ids", statsSettings), ydbClient)
    , DescribingSender(describingSender)
    , SlowDescribingSender(slowDescribingSender)
    , MaxIdsCount(maxIdsCount)
    , DefaultTtl(TDuration::Seconds(defaultTtlSeconds))
{}

void TDescribeIdsProcessor::Process(NHttp::TRequestReply& reply, const TDescribeIdsRequest& request) {
    if (request.Ids.IdsSize() > MaxIdsCount) {
        SendResponse(reply, HTTP_BAD_REQUEST, NHttp::GetSimpleResponse(TStringBuilder() << "Too many ids. Max count = " << MaxIdsCount));
        return;
    }

    try {
        ValidateDescribingExperiment(YdbClient, request.Experiment);
    } catch (const TInvalidDescribingExperimentException& e) {
        SendResponse(reply, HTTP_BAD_REQUEST, NHttp::GetSimpleResponse(e.what()));
        return;
    }

    const auto& userSetId = AddUserSet(
        YdbClient,
        "not_materialized",
        request.Ttl.GetOrElse(DefaultTtl),
        NUserSetHelpers::USER_SET_TYPE.GetNotMaterialized(),
        NUserSetHelpers::USER_SET_STATUS.GetReady()
    );

    ThrowOnError(WaitFor(SaveStats(YdbClient, userSetId, 0, NLab::TUserDataStats(), false)).ValueOrThrow());

    if (IsSlowDescribingMode(request.Mode)) {
        SlowDescribingSender.DescribeIds(userSetId, request.Ids, request.Experiment);
    } else {
        DescribingSender.DescribeIds(userSetId, request.Ids, request.Experiment);
    }

    SendResponse(reply, HTTP_OK, GetResponse(userSetId));
}
