#include "describe_user_set_request_processor.h"

#include <crypta/lib/proto/user_data/user_data.pb.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>
#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/http/format.h>
#include <crypta/lib/native/ydb/executer.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/siberia/bin/common/describing/ydb_requests/ydb_requests.h>
#include <crypta/siberia/bin/common/ydb/parse_utils/parse_utils.h>
#include <crypta/siberia/bin/core/lib/ydb/parse_utils/parse_utils.h>

#include <util/datetime/base.h>

using namespace NCrypta::NSiberia;
using namespace NYdb;
using namespace NTable;

TDescribeUserSetProcessor::TDescribeUserSetProcessor(TYdbClient& ydbClient, TDescribingSender& describingSender, const ::TStats::TSettings& statsSettings)
    : TBase(NLog::GetLog("describe_user_set"), TaggedSingleton<::TStats, decltype(*this)>("processors.describe_user_set", statsSettings), ydbClient)
    , DescribingSender(describingSender)
{}

bool TDescribeUserSetProcessor::TryToClearStats(TUserSetId userSetId) {
    auto result = TryExecuteQueryWithRetries(YdbClient.TableClient, [this, userSetId](NYdb::NTable::TSession session, NYT::TPromise<NYdb::NTable::TDataQueryResult> resultPromise) {
        const auto readResult = GetUserSetStats(session, YdbClient.Database, userSetId, TTxControl::BeginTx(TTxSettings::SerializableRW()));
        if (!readResult.IsSuccess()) {
            return readResult;
        }

        const auto& userSetStats = NYdbCommonParseUtils::ParseUserSetStats(readResult.GetResultSet(0));
        if (userSetStats.Defined() && !userSetStats->Ready) {
            resultPromise.Set(NYT::TError("Describing is already in progress"));
            return readResult;
        }

        auto txControl = TTxControl::Tx(*readResult.GetTransaction()).CommitTx();
        const auto writeResult = SaveStats(session, YdbClient.Database, userSetId, 0, NLab::TUserDataStats(), false, std::move(txControl));

        if (writeResult.IsSuccess()) {
            resultPromise.Set(writeResult);
        }
        return writeResult;
    });

    if (result.IsOK()) {
        ThrowOnError(result.Value());
        return true;
    }

    return false;
}

void TDescribeUserSetProcessor::DoProcess(NHttp::TRequestReply& reply, const TDescribeUserSetRequest& request) {
    if (TryToClearStats(request.UserSetId)) {
        DescribingSender.DescribeUserSet(request.UserSetId);
        SendResponse(reply, HTTP_OK, NHttp::GetSimpleResponse("Send describe command"));
    } else {
        SendResponse(reply, HTTP_CONFLICT, NHttp::GetSimpleResponse("Describing is already in progress"));
    }
}
