#include "get_user_set_stats_request_processor.h"

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/http/format.h>
#include <crypta/lib/native/time/utils.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/siberia/bin/common/describing/ydb_requests/ydb_requests.h>
#include <crypta/siberia/bin/common/proto/stats.pb.h>
#include <crypta/siberia/bin/common/ydb/parse_utils/parse_utils.h>

#include <library/cpp/protobuf/json/proto2json.h>

using namespace NCrypta::NSiberia;
using namespace NYdb;
using namespace NYdb::NTable;


namespace {
    NCrypta::NSiberia::TStats ConvertToStatsProto(const TUserSetStats& userSetStats) {
        NCrypta::NSiberia::TStats res;
        res.MutableUserDataStats()->CopyFrom(userSetStats.UserDataStats);
        res.MutableInfo()->SetProcessedUsersCount(userSetStats.ProcessedUsersCount);
        res.MutableInfo()->SetReady(userSetStats.Ready);
        return res;
    }
}

TGetUserSetStatsProcessor::TGetUserSetStatsProcessor(TYdbClient& ydbClient, const ::TStats::TSettings& statsSettings)
    : TBase(NLog::GetLog("get_user_set_stats"), TaggedSingleton<::TStats, decltype(*this)>("processors.get_user_set_stats", statsSettings), ydbClient)
{}

void TGetUserSetStatsProcessor::DoProcess(NHttp::TRequestReply& reply, const TGetUserSetStatsRequest& request) {
    auto startTime = TInstant::Now();
    const auto result = WaitFor(GetUserSetStats(YdbClient, request.UserSetId)).ValueOrThrow();
    ThrowOnError(result, "Get stats error");
    Stats.Percentile->Add("cryptayt_3678.get_user_set_stats_from_db", MusecsTillNow(startTime));

    startTime = TInstant::Now();
    const auto& userSetStats = NYdbCommonParseUtils::ParseUserSetStats(result.GetResultSet(0));
    Stats.Percentile->Add("cryptayt_3678.parse_user_set_stats", MusecsTillNow(startTime));

    if (userSetStats.Defined()) {
        startTime = TInstant::Now();
        const auto& statsProto = ConvertToStatsProto(*userSetStats);
        Stats.Percentile->Add("cryptayt_3678.convert_to_stats_proto", MusecsTillNow(startTime));

        startTime = TInstant::Now();
        const auto& json = NProtobufJson::Proto2Json(statsProto);
        Stats.Percentile->Add("cryptayt_3678.proto_to_json", MusecsTillNow(startTime));

        startTime = TInstant::Now();
        SendResponse(reply, HTTP_OK, json);
        Stats.Percentile->Add("cryptayt_3678.send_response", MusecsTillNow(startTime));
    } else {
        SendResponse(reply, HTTP_NOT_FOUND, NHttp::GetSimpleResponse("No stats exist"));
    }
}
