#include "worker.h"

#include <crypta/lib/native/http/format.h>
#include <crypta/siberia/bin/core/lib/logic/request_processor_factory.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <library/cpp/http/misc/parsed_request.h>

#include <util/string/builder.h>

using namespace NCrypta;
using namespace NCrypta::NSiberia;
using namespace NCrypta::NHttp;

namespace {
    const TString METRIC_ERRORS_UNEXPECTED = "errors.unexpected";
}


TWorker::TWorker(TRequestProcessorFactory& requestProcessorFactory, const TClientsConfig& clientsConfig, const TStats::TSettings& statsSettings)
    : RequestProcessorFactory(requestProcessorFactory)
    , ClientsConfig(clientsConfig)
    , Stats(TaggedSingleton<TStats, decltype(*this)>("worker", statsSettings))
    , Log(NLog::GetLog("worker"))
{}

void TWorker::ProcessRequest(TRequestReply& reply) {
    try {
        const auto& clientInfo = GetClientInfo(reply.GetClientServiceTicket());

        auto* processor = RequestProcessorFactory.GetProcessor(reply.GetRequestMethod(), reply.GetRequestService(), clientInfo);
        if (processor != nullptr) {
            processor->Process(reply, clientInfo.GetName());
        } else {
            ReplyError(reply, HTTP_NOT_IMPLEMENTED, TStringBuilder() << "Request type " << reply.GetRequestService() << " not supported");
        }
        Y_ENSURE(reply.IsReplied(), "Request not replied");
    } catch (const NTvmAuth::TTvmException& e) {
        ReplyError(reply, HTTP_UNAUTHORIZED, TStringBuilder() << "Invalid TVM ticket: " << e.what());
    } catch (const TRequestProcessorFactory::TNotAllowedException& e) {
        ReplyError(reply, HTTP_FORBIDDEN, TStringBuilder() << "Permission denied: " << e.what());
    } catch (const std::exception& e) {
        Stats.Count->Add(METRIC_ERRORS_UNEXPECTED);
        ReplyError(reply, HTTP_INTERNAL_SERVER_ERROR, TStringBuilder() << "(std::exception) " << e.what());
    } catch (...) {
        Stats.Count->Add(METRIC_ERRORS_UNEXPECTED);
        ReplyError(reply, HTTP_INTERNAL_SERVER_ERROR, "Unknown exception thrown");
    }
}

void TWorker::ReplyError(TRequestReply& reply, HttpCodes httpCode, const TString& message) {
    reply.ReplyError(httpCode, message, Stats, Log);
}

const TClient& TWorker::GetClientInfo(const TMaybe<NTvmAuth::TCheckedServiceTicket>& clientServiceTicket) {
    if (!clientServiceTicket.Defined()) {
        Stats.Count->Add("request.tvm.no_ticket");
        return ClientsConfig.GetAnonymous();
    } else {
        const auto& tvmId = clientServiceTicket->GetSrc();
        const auto& clients = ClientsConfig.GetClients();

        const auto iter = clients.find(tvmId);
        if (iter == clients.end()) {
            Log->error("Unknown client TVM ID = {}", tvmId);
            Stats.Count->Add(TStringBuilder() << "request.tvm.unknown_client_id." << tvmId);
            return ClientsConfig.GetAnonymous();
        } else {
            return iter->second;
        }
    }
}
