#include "add_user_set_db_request.h"

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/ydb/executer.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/lib/native/ydb/join.h>
#include <crypta/siberia/bin/common/data/helpers/user_set_helpers.h>
#include <crypta/siberia/bin/common/ydb/paths/paths.h>
#include <crypta/siberia/bin/core/lib/logic/common/helpers/ids.h>

using namespace NCrypta;
using namespace NCrypta::NSiberia;
using namespace NYdb;
using namespace NYdb::NTable;

namespace {
    void CreateUserSetPath(TYdbClient& ydbClient, TUserSetId userSetId) {
        const auto& path = ydbClient.GetAbsolutePath(GetUserSetDirPath(userSetId));
        const auto result = WaitFor(ydbClient.SchemeClient.MakeDirectory(path)).ValueOrThrow();
        ThrowOnError(result, "Can't create directory for user set");
    }

    static const auto CREATE_TABLE_SETTINGS = TCreateTableSettings().PartitioningPolicy(TPartitioningPolicy().UniformPartitions(2));

    NYdb::TAsyncStatus CreateUsersTable(TYdbClient& ydbClient, TUserSetId userSetId) {
        const auto& path = JoinYdbPath(GetUserSetDirPath(userSetId), "users");
        auto tableDescription = TTableBuilder()
                .AddNullableColumn("id", EPrimitiveType::Uint64)
                .AddNullableColumn("creators", EPrimitiveType::Utf8)
                .AddNullableColumn("status", EPrimitiveType::Utf8)
                .AddNullableColumn("value", EPrimitiveType::Double)
                .SetPrimaryKeyColumn("id")
                .Build();

        return CreateTable(ydbClient, path, std::move(tableDescription), CREATE_TABLE_SETTINGS);
    }

    NYdb::TAsyncStatus CreateUserAttributesTable(TYdbClient& ydbClient, TUserSetId userSetId) {
        const auto& path = JoinYdbPath(GetUserSetDirPath(userSetId), "user_attributes");
        auto tableDescription = TTableBuilder()
                .AddNullableColumn("user_id", EPrimitiveType::Uint64)
                .AddNullableColumn("attribute_key", EPrimitiveType::Utf8)
                .AddNullableColumn("attribute_value", EPrimitiveType::Utf8)
                .AddNullableColumn("ts", EPrimitiveType::Uint64)
                .SetPrimaryKeyColumns({"user_id", "attribute_key", "attribute_value"})
                .Build();

        return CreateTable(ydbClient, path, std::move(tableDescription), CREATE_TABLE_SETTINGS);
    }

    NYdb::TAsyncStatus CreateSegmentsTable(TYdbClient& ydbClient, TUserSetId userSetId) {
        const auto& path = JoinYdbPath(GetUserSetDirPath(userSetId), "segments");
        auto tableDescription = TTableBuilder()
                .AddNullableColumn("id", EPrimitiveType::Uint64)
                .AddNullableColumn("title", EPrimitiveType::Utf8)
                .AddNullableColumn("rule", EPrimitiveType::Utf8)
                .AddNullableColumn("status", EPrimitiveType::Utf8)
                .AddNullableColumn("size", EPrimitiveType::Uint64)
                .AddNullableColumn("creation_ts", EPrimitiveType::Uint64)
                .SetPrimaryKeyColumn("id")
                .Build();

        return CreateTable(ydbClient, path, std::move(tableDescription), CREATE_TABLE_SETTINGS);
    }

    NYdb::TAsyncStatus CreateUserSegmentsTable(TYdbClient& ydbClient, TUserSetId userSetId) {
        const auto& path = JoinYdbPath(GetUserSetDirPath(userSetId), "user_segments");
        auto tableDescription = TTableBuilder()
                .AddNullableColumn("user_id", EPrimitiveType::Uint64)
                .AddNullableColumn("segment_id", EPrimitiveType::Uint64)
                .AddNullableColumn("ts", EPrimitiveType::Uint64)
                .SetPrimaryKeyColumns({"user_id", "segment_id"})
                .Build();

        return CreateTable(ydbClient, path, std::move(tableDescription), CREATE_TABLE_SETTINGS);
    }

    NYdb::TAsyncStatus CreateSegmentUsersTable(TYdbClient& ydbClient, TUserSetId userSetId) {
        const auto& path = JoinYdbPath(GetUserSetDirPath(userSetId), "segment_users");
        auto tableDescription = TTableBuilder()
                .AddNullableColumn("segment_id", EPrimitiveType::Uint64)
                .AddNullableColumn("user_id", EPrimitiveType::Uint64)
                .SetPrimaryKeyColumns({"segment_id", "user_id"})
                .Build();

        return CreateTable(ydbClient, path, std::move(tableDescription), CREATE_TABLE_SETTINGS);
    }

    NYdb::TAsyncStatus CreateSegmentStatsTable(TYdbClient& ydbClient, TUserSetId userSetId) {
        const auto& path = JoinYdbPath(GetUserSetDirPath(userSetId), "segment_stats");
        auto tableDescription = TTableBuilder()
                .AddNullableColumn("segment_id", EPrimitiveType::Uint64)
                .AddNullableColumn("stats", EPrimitiveType::String)
                .SetPrimaryKeyColumn("segment_id")
                .Build();

        return CreateTable(ydbClient, path, std::move(tableDescription), CREATE_TABLE_SETTINGS);
    }

    void CreateUserSetTables(TYdbClient& ydbClient, TUserSetId userSetId) {
        TVector<NYdb::TAsyncStatus> futures = {
            CreateUsersTable(ydbClient, userSetId),
            CreateSegmentsTable(ydbClient, userSetId),
            CreateUserSegmentsTable(ydbClient, userSetId),
            CreateUserAttributesTable(ydbClient, userSetId),
            CreateSegmentUsersTable(ydbClient, userSetId),
            CreateSegmentStatsTable(ydbClient, userSetId)
        };

        WaitForAll(futures).ThrowOnError();
        ThrowOnError(futures, "Can't create tables for user set");
    }
}

NYdb::NTable::TAsyncDataQueryResult NSiberia::ExecuteAddUserSetDbRequest(TYdbClient& ydbClient, const TUserSet& userSet) {
    return ExecuteYdbDynamicQuery<TAddUserSetDbRequest>(ydbClient, "", {.UserSet = userSet});
}


TUserSetId NSiberia::AddUserSet(TYdbClient& ydbClient, const TString& title, TDuration ttl, const TString& type, const TString& status) {
    TUserSet userSet;
    userSet.SetTitle(title);
    userSet.SetExpirationTime((TShiftedClock::Now() + ttl).Seconds());
    userSet.SetType(type);
    userSet.SetStatus(status);

    const auto result = RetryIfConstraintViolation([&]() {
        const auto& newUserSetId = GetNewId();
        userSet.SetId(ToString(newUserSetId));

        if (NUserSetHelpers::IsMaterialized(userSet)) {
            CreateUserSetPath(ydbClient, newUserSetId);
            CreateUserSetTables(ydbClient, newUserSetId);
        }

        return WaitFor(ExecuteAddUserSetDbRequest(ydbClient, userSet)).ValueOrThrow();
    });

    Y_ENSURE(result.Defined(), "Too many attempts to create user set");
    ThrowOnError(*result);

    return FromString<TUserSetId>(userSet.GetId());
}
