#pragma once

#include <crypta/lib/native/ydb/types.h>
#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/siberia/bin/common/data/types.h>
#include <crypta/siberia/bin/common/data/proto/user.pb.h>
#include <crypta/siberia/bin/common/ydb/paths/paths.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <yt/yt/core/actions/future.h>

#include <util/string/subst.h>

namespace NCrypta::NSiberia {
    struct TInsertUsersDbRequest {
        struct TRequestParams {
            const TVector<TUser>& Users;
        };

        static constexpr const char* const Query = R"(
            PRAGMA TablePathPrefix("%s");
            DECLARE $usersData AS "List<Struct<
                id: Uint64,
                status: Utf8>>";

            DECLARE $userAttributesData AS "List<Struct<
                user_id : Uint64,
                attribute_key : Utf8,
                attribute_value: Utf8>>";

            REPLACE INTO {users_table}
            SELECT
                id,
                status
            FROM AS_TABLE($usersData);

            REPLACE INTO {user_attributes_table}
            SELECT
                user_id,
                attribute_key,
                attribute_value
            FROM AS_TABLE($userAttributesData);
        )";

        static TString GetQuery(const TRequestParams&) {
            TString query = Query;
            SubstGlobal(query, "{users_table}", YDB_PATHS.GetUsersTable());
            SubstGlobal(query, "{user_attributes_table}", YDB_PATHS.GetUserAttributesTable());
            return query;
        }

        static NYdb::TParams GetParams(NYdb::TParamsBuilder&& paramsBuilder, const TRequestParams& params) {
            auto& usersParam = paramsBuilder.AddParam("$usersData");
            auto& userAttributesParam = paramsBuilder.AddParam("$userAttributesData");

            usersParam.BeginList();
            userAttributesParam.BeginList();
            for (auto& user : params.Users) {
                const auto& userId = FromString<TUserId>(user.GetId());

                usersParam.AddListItem()
                    .BeginStruct()
                    .AddMember("id").Uint64(userId)
                    .AddMember("status").Utf8(user.GetInfo().GetStatus())
                    .EndStruct();

                for (const auto& [key, values] : user.GetInfo().GetAttributes()) {
                    for (const auto& value : values.GetValues()) {
                        userAttributesParam.AddListItem()
                            .BeginStruct()
                            .AddMember("user_id").Uint64(userId)
                            .AddMember("attribute_key").Utf8(key)
                            .AddMember("attribute_value").Utf8(value)
                            .EndStruct();
                    }
                }
            }
            usersParam.EndList().Build();
            userAttributesParam.EndList().Build();

            return paramsBuilder.Build();
        }
    };

    NYdb::NTable::TAsyncDataQueryResult ExecuteInsertUsersDbRequest(TYdbClient& ydbClient, TUserSetId userSetId, const TVector<TUser>& users);
    void InsertUsers(TYdbClient& ydbClient, TUserSetId userSetId, const TVector<TUser>& users);
}
