import datetime
import logging
import os

from crypta.lib.python import templater
from crypta.lib.python.yt import yt_helpers
from crypta.profile.lib import date_helpers
from crypta.siberia.bin.custom_audience.apps_clustering.lib.utils import (
    config,
    fields,
)
from crypta.siberia.bin.custom_audience.lib.python.clustering import utils as clustering_utils


logger = logging.getLogger(__name__)

get_userid_clusterid_table_template = """
$app2id_table = (
    SELECT
        clustering_table.{{fields.id_type}} AS {{fields.id_type}},
        clustering_table.{{fields.cluster_id}} AS {{fields.cluster_id}},
        devid_by_app_table.id AS id,
    FROM `{{clustering_table}}` AS clustering_table
    INNER JOIN `{{devid_by_app_table}}` AS devid_by_app_table
    ON clustering_table.{{fields.name}} == devid_by_app_table.app
        AND clustering_table.{{fields.id_type}} == devid_by_app_table.id_type
);

INSERT INTO `{{userid_clusterid_table}}`
WITH TRUNCATE

SELECT
    id_to_crypta_id_table.crypta_id AS id,
    'crypta_id' AS id_type,
    app2id_table.{{fields.cluster_id}} AS {{fields.cluster_id}},
FROM $app2id_table AS app2id_table
INNER JOIN `{{id_to_crypta_id_table}}` AS id_to_crypta_id_table
ON app2id_table.{{fields.id_type}} == id_to_crypta_id_table.id_type
    AND app2id_table.id == id_to_crypta_id_table.id
ORDER BY id, id_type;
"""


def get_userid_clusterid_table(yt_client, transaction, yql_client, clustering_table, userid_clusterid_table):
    yql_client.execute(
        query=templater.render_template(
            get_userid_clusterid_table_template,
            vars={
                'clustering_table': clustering_table,
                'devid_by_app_table': config.DEVID_BY_APP_TABLE,
                'id_to_crypta_id_table': config.ID_TO_CRYPTA_ID_TABLE,
                'userid_clusterid_table': userid_clusterid_table,
                'fields': fields,
            },
        ),
        transaction=str(transaction.transaction_id),
        title='YQL get_userid_clusterid_table',
    )
    yt_helpers.set_ttl(
        table=userid_clusterid_table,
        ttl_timedelta=datetime.timedelta(days=config.DAYS_TO_KEEP_USERID_CLUSTERID),
        yt_client=yt_client,
    )


def update(yt_client, yql_client):
    today = date_helpers.get_today_date_string()

    with yt_client.Transaction() as transaction:
        userid_clusterid_table = os.path.join(config.CRYPTAID_CLUSTERID_DIR, today)
        get_userid_clusterid_table(
            yt_client,
            transaction=transaction,
            yql_client=yql_client,
            clustering_table=config.CLUSTERING_TABLE,
            userid_clusterid_table=userid_clusterid_table,
        )

        centroids_last_update_planned_date = yt_client.get_attribute(
            path=config.CENTROIDS_TABLE,
            attribute='last_update_planned_date',
        )

        clustering_utils.update_centroids_users_count(
            transaction=transaction,
            yql_client=yql_client,
            userid_clusterid_table=userid_clusterid_table,
            config=config,
            fields=fields,
        )

        for attribute, value in [
            ('last_update_planned_date', centroids_last_update_planned_date),
            ('generate_date', today),
        ]:
            yt_client.set_attribute(
                path=config.CENTROIDS_TABLE,
                attribute=attribute,
                value=value,
            )
