import json

from crypta.lib.python import templater
from crypta.siberia.bin.custom_audience.apps_clustering.lib.utils import (
    config,
    fields,
    yt_schemas,
)
from crypta.siberia.bin.custom_audience.lib.python.clustering import utils
from crypta.lib.python.yt import yt_helpers


get_apps_from_stores_template = """
$from_store_table = (
    SELECT
        BundleId AS {{fields.name}},
        CASE SourceID
            WHEN 1 THEN 'gaid'
            WHEN 2 THEN 'idfa'
            ELSE NULL
        END AS {{fields.id_type}},
        SOME(AppHashMd5) AS {{fields.md5_hash}},
        Cast(SearchRequest::NormalizeBert(SOME(Description)) as String) ?? "" AS {{fields.description}},
    FROM `{{app_data_table}}`
    WHERE RegionName == 'ru' AND SourceID IS NOT NULL
    GROUP BY BundleId, SourceID
);

{% if only_new_apps %}
$from_store_table = (
    SELECT *
    FROM $from_store_table AS from_store_table
    LEFT ONLY JOIN `{{apps_clustering_table}}` AS apps_clustering_table
    USING ({{fields.name}}, {{fields.id_type}})
);
{% endif %}

$apps_with_devids_table = (
    SELECT
        from_store_table.*,
        devid_by_app_table.id AS id,
    FROM $from_store_table AS from_store_table
    INNER JOIN `{{devid_by_app_table}}` AS devid_by_app_table
    ON from_store_table.{{fields.name}} == devid_by_app_table.app
        AND from_store_table.{{fields.id_type}} == devid_by_app_table.id_type
);

$apps_with_devids_count_table = (
    SELECT *
    FROM  (
        SELECT
            {{fields.name}},
            {{fields.id_type}},
            SOME({{fields.md5_hash}}) AS {{fields.md5_hash}},
            COUNT(DISTINCT id) AS {{fields.users_count}},
            SOME({{fields.description}}) AS {{fields.description}},
        FROM $apps_with_devids_table
        GROUP BY {{fields.name}}, {{fields.id_type}}
    )
    WHERE {{fields.users_count}} >= {{min_devids_count}}
);

$max_app_id = (
    SELECT
        COALESCE(MAX(app_id), 0)
    FROM `{{app_dict_table}}`
);

$new_app_dict_table = (
    SELECT
        bundle_id,
        id_type,
        app_id,
    FROM `{{app_dict_table}}`
UNION ALL
    SELECT
        {{fields.name}} as bundle_id,
        {{fields.id_type}} as id_type,
        $max_app_id + ROW_NUMBER() OVER w AS app_id,
    FROM $apps_with_devids_count_table AS apps_with_devids_count_table
    LEFT ONLY JOIN `{{app_dict_table}}` AS app_dict_table
    USING ({{fields.name}}, {{fields.id_type}})
    WINDOW w AS (ORDER BY apps_with_devids_count_table.{{fields.users_count}} DESC)
);


INSERT INTO `{{app_dict_table}}`
WITH TRUNCATE

SELECT *
FROM $new_app_dict_table;


INSERT INTO `{{apps_from_stores_table}}`
WITH TRUNCATE

SELECT
    apps_with_devids_count_table.*,
    new_app_dict_table.{{fields.id}} AS {{fields.id}}
FROM $apps_with_devids_count_table AS apps_with_devids_count_table
INNER JOIN $new_app_dict_table AS new_app_dict_table
USING ({{fields.name}}, {{fields.id_type}})
"""


def get_apps_from_stores(yt_client, yql_client, nv_params, output):
    stages_dir = config.CLUSTERING_UPDATE_STAGES_DIR if not nv_params['only_new_apps'] else config.APPS_UPDATE_STAGES_DIR

    with yt_client.Transaction() as transaction:
        yql_client.execute(
            query=templater.render_template(
                get_apps_from_stores_template,
                vars={
                    'app_data_table': config.APP_DATA_TABLE,
                    'only_new_apps': nv_params['only_new_apps'],
                    'apps_clustering_table': config.CLUSTERING_TABLE,
                    'devid_by_app_table': config.DEVID_BY_APP_TABLE,
                    'app_dict_table': config.APP_DICT_TABLE,
                    'min_devids_count': config.MIN_DEVIDS_COUNT,
                    'apps_from_stores_table': utils.get_segments_with_info_table_path(stages_dir),
                    'fields': fields,
                },
            ),
            transaction=str(transaction.transaction_id),
            title='YQL get_apps_from_stores',
        )

        yt_helpers.create_empty_table(
            yt_client=yt_client,
            path=utils.get_segments_vectors_with_info_table_path(stages_dir),
            schema=yt_schemas.get_segments_vectors_with_info_schema(),
        )

        with open(output, 'w') as output_file:
            json.dump({
                'segments_with_info_table': utils.get_segments_with_info_table_path(stages_dir),
                'segments_vectors_with_info_table': utils.get_segments_vectors_with_info_table_path(stages_dir),
                'description_field': fields.description,
                'vector_field': fields.vector,
            }, output_file)
