#include "custom_audience_server.h"

#include "custom_audience_service_impl.h"

#include <crypta/lib/native/grpc/keepalive/keepalive.h>
#include <crypta/lib/native/grpc/logging/logging_interceptor_factory.h>
#include <crypta/lib/native/grpc/stats/stats_interceptor_factory.h>

#include <util/string/cast.h>

using namespace NCrypta::NSiberia::NCustomAudience;

TCustomAudienceServer::TCustomAudienceServer(
    const TServiceConfig& config,
    const TDb& db,
    const TStats::TSettings& statsSettings
)
    : Port(config.GetPort())
    , MaxReceiveMessageSize(config.GetMaxReceiveMessageSize())
    , Service(MakeHolder<TCustomAudienceServiceImpl>(config.GetThreadCount(), db, config.GetDescribeThreshold(), statsSettings))
    , Log(NLog::GetLog("custom_audience_server"))
    , StatsSettings(statsSettings)
{
}

void TCustomAudienceServer::Start() {
    const auto& hostPort = TString("0.0.0.0:") + ToString(Port);

    grpc::ServerBuilder builder;
    builder.AddListeningPort(hostPort, grpc::InsecureServerCredentials());
    builder.RegisterService(Service.Get());
    builder.SetMaxReceiveMessageSize(MaxReceiveMessageSize);

    NCrypta::NGrpc::EnableServerKeepalive(builder);

    std::vector<std::unique_ptr<grpc::experimental::ServerInterceptorFactoryInterface>> creators;

    creators.push_back(std::make_unique<NCrypta::NGrpc::TLoggingInterceptorFactory>(
        [&](TString&& line){ Log->info(std::move(line)); },
        Log
        ));
    creators.push_back(std::make_unique<NCrypta::NGrpc::TStatsInterceptorFactory>(StatsSettings));

    builder.experimental().SetInterceptorCreators(std::move(creators));

    Server = builder.BuildAndStart();

    if (!Server) {
        ythrow yexception() << "Can't start server";
    }

    Log->info("TCustomAudienceServer is listening on {}", hostPort);
}

TCustomAudienceServer::~TCustomAudienceServer() {
    Log->info("Shutting down the server...");
    Server->Shutdown();
    Server->Wait();
    Log->info("Done");
}
