#include "get_sorted_segments.h"

#include <library/cpp/regex/pcre/pcre.h>

#include <util/charset/utf8.h>

namespace {
    static const TVector<TString> LANGUAGES = {"ru", "en"};

    int GetPriority(const TString& requestText, const NCrypta::NSiberia::NCustomAudience::NSuggester::TDatabaseState::TSegmentItem::TLangSpecificInfo& info) {
        static const int TextInName = 1;
        static const int PrefixInDescription = 2;
        static const int PrefixInName = 4;

        int priority = 0;
        const auto& name = ToLowerUTF8(info.Name);
        const NPcre::TPcre prefix(("\\b" + requestText).data(), NPcre::EOptimize::None, PCRE_UTF8 | PCRE_UCP | PCRE_CASELESS);

        if (name.Contains(requestText)) {
            priority |= TextInName;
        }
        if (prefix.Matches(name)) {
            priority |= PrefixInName;
        }
        if (prefix.Matches(info.Description)) {
            priority |= PrefixInDescription;
        }

        return priority;
    }

    struct TItemWithPriority {
        NCrypta::NSiberia::NCustomAudience::NSuggester::TItem Item;
        int Priority;
    };
}

::google::protobuf::RepeatedPtrField<NCrypta::NSiberia::NCustomAudience::NSuggester::TItem> NCrypta::NSiberia::NCustomAudience::NSuggester::GetSortedSegments(
    const TVector<TDatabaseState::TSegmentItem>& segments,
    const TString& requestText,
    const ui64 maxItems,
    NRawData::TExport::ECampaignType campaignType,
    const TString& locale
) {
    ::google::protobuf::RepeatedPtrField<TItem> result;
    TVector<TItemWithPriority> items;
    items.reserve(segments.size());

    for (const auto& segmentItem: segments) {
        if (campaignType != NRawData::TExport::all && !segmentItem.CampaignTypes.contains(campaignType)) {
            continue;
        }

        for (const auto& language : LANGUAGES) {
            if (!locale.empty() && locale != language) {
                continue;
            }

            const auto& info = segmentItem.LangInfo.at(language);
            if (info.SearchString.Contains(requestText)) {
                TItemWithPriority itemWithPriority;
                itemWithPriority.Priority = GetPriority(requestText, info);
                itemWithPriority.Item.SetType(segmentItem.Type);
                itemWithPriority.Item.SetText(info.Name);
                itemWithPriority.Item.SetDescription(info.Description);
                itemWithPriority.Item.MutableExports()->CopyFrom(segmentItem.Exports);
                items.push_back(std::move(itemWithPriority));
                break;
            }
        }
    }

    auto middle = static_cast<ui64>(items.size()) > maxItems ? items.begin() + maxItems : items.end();
    std::partial_sort(items.begin(), middle, items.end(), [](const auto& left, const auto& right) {
        return left.Priority == right.Priority ? left.Item.GetText() < right.Item.GetText() : left.Priority > right.Priority;
    });

    result.Reserve(maxItems);
    for (auto it = items.begin(); it < middle; ++it) {
        result.Add(std::move(it->Item));
    }

    return result;
}
