#include "suggester_service_impl.h"

#include "get_sorted_segments.h"

#include <crypta/lib/native/time/scope_timer.h>

#include <util/charset/utf8.h>
#include <util/generic/refcount.h>

using namespace NCrypta::NSiberia::NCustomAudience::NSuggester;

namespace {
    static const TString EMPTY_SUGGEST_QUERY = "ресторан";

    static const THashMap<TString, TString> APP_LOCALES_REMAPPING = {
        {"en", "us"},
        {"ru", "ru"},
        {"", ""},
    };

    void AddHost(const TString& host, const ui64 id, ::google::protobuf::RepeatedPtrField<TItem>& items) {
        auto& item = *items.Add();
        item.SetType("host");
        item.SetText(host);
        item.SetHostId(id);
    }

    void AddApp(const TDatabaseState::TApps::TAppInfo& appInfo, ::google::protobuf::RepeatedPtrField<TItem>& items) {
        auto& item = *items.Add();
        item.MutableApp()->CopyFrom(appInfo.Raw);
    }

    bool AppInfoMatches(const TDatabaseState::TApps::TAppInfo& appInfo, const TStringBuf& searchString, const TStringBuf& locale) {
        if (appInfo.NormalizedBundleId.Contains(searchString)) {
            return true;
        }
        for (const auto& [lang, title] : appInfo.NormalizedTitles) {
            if ((locale.empty() || locale == lang) && title.Contains(searchString)) {
                return true;
            }
        }
        return false;
    }

    ::google::protobuf::RepeatedPtrField<TItem> GetSortedApps(
        const TVector<TDatabaseState::TApps::TAppInfo>& appInfos,
        const TStringBuf& searchString,
        const int itemsToFillCount,
        const TStringBuf& locale
    ) {
        ::google::protobuf::RepeatedPtrField<TItem> result;
        result.Reserve(itemsToFillCount);

        for (const auto& appInfo : appInfos) {
            if (AppInfoMatches(appInfo, searchString, locale)) {
                AddApp(appInfo, result);

                if (result.size() == itemsToFillCount) {
                    break;
                }
            }
        }

        return result;
    }
}

TSuggesterServiceImpl::TSuggesterServiceImpl(const TDatabase& database, const TStats::TSettings& statsSettings)
    : Database(database)
    , Log(NLog::GetLog("suggester_service"))
    , Stats(TaggedSingleton<TStats, decltype(*this)>("suggester_server", statsSettings))
{}

grpc::Status TSuggesterServiceImpl::Ping(grpc::ServerContext*, const google::protobuf::Empty*, TPingResponse* response) {
    response->SetMessage("OK");
    return grpc::Status::OK;
}

grpc::Status TSuggesterServiceImpl::Ready(grpc::ServerContext*, const google::protobuf::Empty*, TPingResponse* response) {
    if (!Database.IsReady()) {
        return grpc::Status(grpc::StatusCode::UNAVAILABLE, "Service is not ready");
    }

    response->SetMessage("OK");
    return grpc::Status::OK;
}

grpc::Status TSuggesterServiceImpl::Suggest(grpc::ServerContext*, const TSuggestRequest* request, TSuggestResponse* response) {
    try {
        if (!Database.IsReady()) {
            return grpc::Status(grpc::StatusCode::UNAVAILABLE, "Service is not ready");
        }
        const auto& state = Database.GetState();

        size_t itemsToFillCount = request->GetLimit();
        if (itemsToFillCount == 0) {
            return grpc::Status::OK;
        }

        const auto& requestText = request->GetText().empty() ? EMPTY_SUGGEST_QUERY : ToLowerUTF8(request->GetText());

        auto items = GetSortedSegments(state->GetSegments(), requestText, itemsToFillCount, request->GetCampaignType(), request->GetLocale());
        itemsToFillCount -= items.size();
        response->MutableItems()->Swap(&items);

        if (itemsToFillCount != 0 && request->GetFlags().GetAppsEnabled()) {
            const auto& apps = state->GetApps();
            auto items = GetSortedApps(apps.SortedByCount, requestText, itemsToFillCount, APP_LOCALES_REMAPPING.at(request->GetLocale()));
            itemsToFillCount -= items.size();

            response->MutableItems()->Add(items.begin(), items.end());
        }

        if (itemsToFillCount != 0) {
            const auto& tails = state->GetHostsTrie().FindTails(requestText.begin(), requestText.size());

            TVector<TDatabaseState::TTrie::TValueType> sortedHosts(std::min(itemsToFillCount, tails.Size()));
            std::partial_sort_copy(tails.begin(), tails.end(), sortedHosts.begin(), sortedHosts.end(), [](const auto& left, const auto& right) {
                return left.second.first > right.second.first;
            });

            for (const auto& host : sortedHosts) {
                AddHost(requestText + host.first, host.second.second, *response->MutableItems());
            }
        }

        return grpc::Status::OK;
    } catch (const yexception& e) {
        return grpc::Status(grpc::StatusCode::INTERNAL, e.what());
    } catch (const std::exception& e) {
        return grpc::Status(grpc::StatusCode::INTERNAL, e.what());
    } catch (...) {
        return grpc::Status(grpc::StatusCode::INTERNAL, "Unknown exception");
    }
}

grpc::Status TSuggesterServiceImpl::GetHosts(grpc::ServerContext*, const TGetHostsRequest* request, TGetHostsResponse* response) {
    try {
        if (!Database.IsReady()) {
            return grpc::Status(grpc::StatusCode::UNAVAILABLE, "Service is not ready");
        }
        const auto& state = Database.GetState();

        for (const auto& id : request->GetHostIds()) {
            const auto& host = state->GetHostById(id);
            if (host.Defined()) {
                AddHost(*host, id, *response->MutableItems());
            }
        }

        return grpc::Status::OK;
    } catch (const yexception& e) {
        return grpc::Status(grpc::StatusCode::INTERNAL, e.what());
    } catch (const std::exception& e) {
        return grpc::Status(grpc::StatusCode::INTERNAL, e.what());
    } catch (...) {
        return grpc::Status(grpc::StatusCode::INTERNAL, "Unknown exception");
    }
}

grpc::Status TSuggesterServiceImpl::GetApps(grpc::ServerContext*, const TGetAppsRequest* request, TGetAppsResponse* response) {
    try {
        if (!Database.IsReady()) {
            return grpc::Status(grpc::StatusCode::UNAVAILABLE, "Service is not ready");
        }
        const auto& state = Database.GetState();

        for (const auto& id : request->GetAppIds()) {
            const auto& appInfo = state->GetAppInfoById(id);
            if (appInfo.Defined()) {
                AddApp(*appInfo, *response->MutableItems());
            }
        }

        return grpc::Status::OK;
    } catch (const yexception& e) {
        return grpc::Status(grpc::StatusCode::INTERNAL, e.what());
    } catch (const std::exception& e) {
        return grpc::Status(grpc::StatusCode::INTERNAL, e.what());
    } catch (...) {
        return grpc::Status(grpc::StatusCode::INTERNAL, "Unknown exception");
    }
}
