#include <crypta/lib/native/cmd_args/parse_pb_options.h>
#include <crypta/lib/native/event_processing/worker.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/log/utils/log_exception.h>
#include <crypta/lib/native/log/wrappers/pqlib/pq_spdlog_wrapper.h>
#include <crypta/lib/native/pqlib/consumer.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/reader.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/graphite/graphite_stats_sink.h>
#include <crypta/lib/native/stats/log/log_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yaml/parse_yaml_file.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/native/yaml/ytree_node/utils.h>
#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/lib/native/yt/yson/utils.h>
#include <crypta/siberia/bin/describer/bin/cmd_args.pb.h>
#include <crypta/siberia/bin/describer/lib/config/config.pb.h>
#include <crypta/siberia/bin/describer/lib/processor_factory.h>

#include <library/cpp/json/json_reader.h>
#include <yt/yt/core/concurrency/thread_pool.h>
#include <yt/yt/core/misc/shutdown.h>

#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/stream/str.h>

using namespace NCrypta;
using namespace NCrypta::NLog;
using namespace NCrypta::NSiberia::NDescriber;

namespace {
    TConfig ParseArgs(int argc, const char** argv) {
        const auto& options = ParsePbOptions<TCmdArgs>(argc, argv);
        auto yamlConfig = ParseYamlFile(options.GetConfigFile());

        return Yaml2Proto<TConfig>(yamlConfig);
    }

    TStats::TSettings GetStatsSettings() {
        return {
            .PercentMax = 10 * 60 * 1000000
        };
    }
}

int main(int argc, const char** argv) {
    TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

    try {
        TShiftedClock::FreezeTimestampFromEnv();

        auto config = ParseArgs(argc, argv);

        RegisterLogs(config.GetLogs());

        auto statsRegistry = Singleton<TStatsRegistry>();
        TStatsWriter statsWriter(*statsRegistry, TDuration::Minutes(1));
        if (config.HasGraphite()) {
            statsWriter.AddSink(MakeHolder<TGraphiteStatsSink>(config.GetGraphite()));
        }
        if (config.HasSolomon()) {
            statsWriter.AddSink(MakeHolder<TSolomonStatsSink>(config.GetSolomon()));
        }
        statsWriter.AddSink(MakeHolder<TLogStatsSink>("graphite"));
        statsWriter.Start();

        const auto& statsSettings = GetStatsSettings();

        TStats pqlibStats("pqlib", statsSettings);
        const auto& logbrokerConfig = config.GetLogbroker();
        auto pqLibLogger = MakeIntrusive<NCrypta::NLog::TPqSpdlogWrapper>(logbrokerConfig.GetPqlibLogName());
        NPQ::TConsumer consumer(
            NPQ::NPQLib::Create(logbrokerConfig.GetPqlib()),
            pqLibLogger,
            logbrokerConfig.GetConsumer(),
            NPQ::NCredentialsProvider::Create(logbrokerConfig.GetCredentials(), pqLibLogger),
            pqlibStats
        );

        TYdbClient ydbClient(config.GetYdb());

        TStats workerStats("worker", statsSettings);
        TProcessorFactory processorFactory(ydbClient, config.GetDescribingBatchSize(), config.GetStatsUpdateThreshold(), 100 * 1000, workerStats);

        NPQ::TCookieQueue cookiesToCommit;
        auto worker = NYT::New<NEventProcessing::TWorker>(processorFactory, cookiesToCommit, workerStats);

        auto threadPool = NYT::New<NYT::NConcurrency::TThreadPool>(config.GetWorker().GetThreads(), "worker");

        NPQ::TReader reader(
            TDuration::MilliSeconds(config.GetReader().GetReadTimeoutMs()),
            consumer,
            cookiesToCommit,
            [&worker, &threadPool] (auto&& readResult) {
                threadPool->GetInvoker()->Invoke(BIND([&worker, readResult = std::move(readResult)] () mutable {
                    worker->Process(std::move(readResult));
                }));
            },
            NLog::GetLog(logbrokerConfig.GetPqlibLogName())
        );
        reader.SetOnFinishHandler([&stopSignalWaiter] () {
            stopSignalWaiter.Signal();
        });

        GetLog("main")->info("Start");
        reader.Start();

        stopSignalWaiter.Wait();

        GetLog("main")->info("Stop");
    } catch (const yexception& e) {
        LogException(e.what());
        return 1;
    } catch (const std::exception& e) {
        LogException(e.what());
        return 1;
    } catch (...) {
        NLog::LogException("Unknown exception");
        return 1;
    }

    Cerr << "Shutting down YT library" << Endl;
    NYT::Shutdown();

    Cerr << "Exiting" << Endl;
    return 0;
}
