#include "describe_processor.h"

#include "ydb_requests.h"

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/ydb/executer.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>
#include <crypta/siberia/bin/common/describing/ydb_requests/ydb_requests.h>
#include <crypta/siberia/bin/common/ydb/parse_utils/parse_utils.h>

using namespace NCrypta;
using namespace NCrypta::NSiberia::NDescriber;

TDescribeProcessor::TDescribeProcessor(
    TInstant commandTimestamp,
    TYdbClient& ydbClient,
    size_t describingBatchSize,
    size_t statsUpdateThreshold,
    ::TStats& stats
)
    : TBase(ydbClient, stats)
    , DescribingBatchSize(describingBatchSize)
    , StatsUpdateThreshold(statsUpdateThreshold)
    , CommandTimestamp(commandTimestamp)
{}

bool TDescribeProcessor::ProcessUnsafe() {
    const auto& isOk = ProcessCommand();
    if (isOk) {
        const auto& uniqueIdsCount = ProcessedCryptaIds.size();
        Stats.Percentile->Add("crypta_id.total", TotalCryptaIdsCount);
        Stats.Percentile->Add("crypta_id.unique", uniqueIdsCount);

        if (TotalCryptaIdsCount != 0) {
            Stats.Percentile->Add("crypta_id.unique_rate", (uniqueIdsCount * 100) / TotalCryptaIdsCount);
        }

        Stats.Percentile->Add("command.latency", (TShiftedClock::Now() - CommandTimestamp).Seconds());
    }
    return isOk;
}

void TDescribeProcessor::UpdateStats(TUserSetId userSetId, ui64 processedUsersCount, bool ready) {
    const auto result = ExecuteQueryWithRetries(YdbClient.TableClient, [this, processedUsersCount, userSetId, ready](NYdb::NTable::TSession session, NYT::TPromise<NYdb::NTable::TDataQueryResult> resultPromise) {
        const auto getStatsResult = GetUserSetStats(session, YdbClient.Database, userSetId, NYdb::NTable::TTxControl::BeginTx(NYdb::NTable::TTxSettings::SerializableRW()));
        if (!getStatsResult.IsSuccess()) {
            return getStatsResult;
        }

        const auto& stats = NYdbCommonParseUtils::ParseUserSetStats(getStatsResult.GetResultSet(0));
        NLab::TUserDataStatsAggregator<> tmpStatsAggregator(*StatsAggregator);
        if (stats.Defined()) {
            tmpStatsAggregator.UpdateWith(stats->UserDataStats);
        }

        NLab::TUserDataStats userDataStats;
        tmpStatsAggregator.MergeInto(userDataStats);

        auto txControl = NYdb::NTable::TTxControl::Tx(*getStatsResult.GetTransaction()).CommitTx();
        const auto saveResult = SaveStats(session, YdbClient.Database, userSetId, processedUsersCount, userDataStats, ready, std::move(txControl));
        if (saveResult.IsSuccess()) {
            resultPromise.Set(saveResult);
        }

        return saveResult;
    });

    ThrowOnError(result, AddMessagePrefix(userSetId, "Update stats error"));

    StatsAggregator = MakeHolder<NLab::TUserDataStatsAggregator<>>();
    StoredProcessedUsersCount = processedUsersCount;
}

void TDescribeProcessor::UpdateStatsIfNecessary(TUserSetId userSetId, ui64 processedUsersCount) {
    if (processedUsersCount - StoredProcessedUsersCount >= StatsUpdateThreshold) {
        UpdateStats(userSetId, processedUsersCount);
    }
}

void TDescribeProcessor::DeleteOldStats(TUserSetId userSetId) {
    LogInfo(userSetId, "Delete old stats", userSetId);
    ThrowOnError(WaitFor(SaveStats(YdbClient, userSetId, 0, NLab::TUserDataStats(), false)).Value(), AddMessagePrefix(userSetId, "Save stats error"));
}

void TDescribeProcessor::AggregateStats(const NYdb::TResultSet& resultSet) {
    NYdb::TResultSetParser parser(resultSet);
    while (parser.TryNextRow()) {
        NYdbCommonParseUtils::ParseUserDataStats(UserDataStats, parser, "stats");
        StatsAggregator->UpdateWith(UserDataStats);
    }
}

void TDescribeProcessor::DescribeCryptaIds(TUserSetId userSetId, const TString& userDataTablePath, const NYdb::TResultSet& resultSet) {
    const auto& cryptaIds = ParseUnprocessedCryptaIds(resultSet);
    DescribeCryptaIds(userSetId, userDataTablePath, cryptaIds);
}

TVector<ui64> TDescribeProcessor::ParseUnprocessedCryptaIds(const NYdb::TResultSet& resultSet) {
    TVector<ui64> unprocessedCryptaIds;
    unprocessedCryptaIds.reserve(DescribingBatchSize);

    NYdb::TResultSetParser parser(resultSet);
    while (parser.TryNextRow()) {
        const auto& cryptaId = *parser.ColumnParser("crypta_id").GetOptionalUint64();
        const auto& [it, actuallyInserted] = ProcessedCryptaIds.insert(cryptaId);

        if (actuallyInserted) {
            unprocessedCryptaIds.push_back(cryptaId);
        }

        ++TotalCryptaIdsCount;
    }

    return unprocessedCryptaIds;
}

void TDescribeProcessor::DescribeCryptaIds(TUserSetId userSetId, const TString& userDataTablePath, const TVector<ui64>& cryptaIds) {
    if (!cryptaIds.empty()) {
        const auto result = WaitFor(::DescribeCryptaIds(YdbClient, cryptaIds, userDataTablePath)).ValueOrThrow();
        ThrowOnError(result, AddMessagePrefix(userSetId, "Describe crypta ids error"));
        AggregateStats(result.GetResultSet(0));
    }
}
