#pragma once

#include <crypta/lab/lib/native/stats.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/stats/stats.h>
#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/siberia/bin/common/data/types.h>
#include <crypta/siberia/bin/common/describing/proto/describe_ids_cmd.pb.h>
#include <crypta/siberia/bin/common/event_processing/processor.h>

#include <util/generic/hash_set.h>

namespace NCrypta::NSiberia::NDescriber {
    class TDescribeProcessor : public TProcessor {
        using TBase = TProcessor;

    public:
        TDescribeProcessor(
            TInstant commandTimestamp,
            TYdbClient& ydbClient,
            size_t describingBatchSize,
            size_t statsUpdateThreshold,
            ::TStats& stats
        );

    protected:
        void AggregateStats(const NYdb::TResultSet& resultSet);
        void UpdateStats(TUserSetId userSetId, ui64 processedUsersCount, bool ready = false);
        void UpdateStatsIfNecessary(TUserSetId userSetId, ui64 processedUsersCount);
        void DeleteOldStats(TUserSetId userSetId);
        void DescribeCryptaIds(TUserSetId userSetId, const TString& userDataTablePath, const NYdb::TResultSet& resultSet);

        virtual TString AddMessagePrefix(TUserSetId userSetId, const TString& message) = 0;

        template <typename... Args>
        void LogMessage(spdlog::level::level_enum level, TUserSetId userSetId, const char* fmt, const Args&... args) {
            Log->log(level, AddMessagePrefix(userSetId, fmt).c_str(), args...);
        }

        template <typename... Args>
        void LogInfo(TUserSetId userSetId, const char* fmt, const Args&... args) {
            LogMessage(spdlog::level::level_enum::info, userSetId, fmt, args...);
        }

        template <typename... Args>
        void LogError(TUserSetId userSetId, const char* fmt, const Args&... args) {
            LogMessage(spdlog::level::level_enum::err, userSetId, fmt, args...);
        }

        size_t DescribingBatchSize = 0;
        size_t StatsUpdateThreshold = 0;
        TInstant CommandTimestamp;
        THashSet<ui64> ProcessedCryptaIds;
        size_t TotalCryptaIdsCount = 0;
        size_t StoredProcessedUsersCount = 0;
        NLab::TUserDataStats UserDataStats;
        THolder<NLab::TUserDataStatsAggregator<>> StatsAggregator = MakeHolder<NLab::TUserDataStatsAggregator<>>();

    private:
        bool ProcessUnsafe() override;
        virtual bool ProcessCommand() = 0;

        TVector<ui64> ParseUnprocessedCryptaIds(const NYdb::TResultSet& resultSet);
        void DescribeCryptaIds(TUserSetId userSetId, const TString& userDataTablePath, const TVector<ui64>& cryptaIds);
    };
}
