#include "describe_user_set_processor.h"

#include "ydb_requests.h"

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/lib/native/ydb/ydb_error_exception.h>
#include <crypta/siberia/bin/common/describing/paths/paths.h>
#include <crypta/siberia/bin/common/ydb/parse_utils/parse_utils.h>
#include <crypta/siberia/bin/common/ydb/requests/user_set_exists_db_request.h>
#include <crypta/siberia/bin/common/ydb/tables/user_data_stats_table_fields.h>
#include <crypta/siberia/bin/common/ydb/tables/user_set_stats_fields.h>

using namespace NCrypta::NSiberia;
using namespace NCrypta::NSiberia::NDescriber;

namespace {
    struct TUsersInfo {
        TMaybe<ui64> LastUserId;
        ui64 ProcessedUsersCount = 0;
    };

    TUsersInfo ParseUsersInfo(const NYdb::TResultSet& resultSet) {
        NYdb::TResultSetParser parser(resultSet);
        Y_ENSURE(parser.TryNextRow(), "No user info");
        return {
            .LastUserId = parser.ColumnParser("last_user_id").GetOptionalUint64(),
            .ProcessedUsersCount = parser.ColumnParser("count").GetUint64()
        };
    }
}

TDescribeUserSetProcessor::TDescribeUserSetProcessor(
    TDescribeUserSetCmd command,
    TInstant commandTimestamp,
    TYdbClient& ydbClient,
    size_t describingBatchSize,
    size_t statsUpdateThreshold,
    size_t maxUserProcessedCount,
    ::TStats& stats
)
    : TBase(commandTimestamp, ydbClient, describingBatchSize, statsUpdateThreshold, stats)
    , Command(std::move(command))
    , MaxUserProcessedCount(maxUserProcessedCount)
{}

bool TDescribeUserSetProcessor::ProcessCommand() {
    TScopeTimer timer(Stats.Percentile, "describe_user_set.timing.user_set_processing");
    const auto& userSetId = Command.GetUserSetId();

    auto logWithInfo = [this, userSetId](const char* msg, const auto&... params) {
        LogInfo(userSetId, msg, params...);
    };

    if (!IsUserSetExists(YdbClient, userSetId)) {
        logWithInfo("Skip describing because user set no longer exists");
        Stats.Count->Add("describe_user_set.count.absent_user_sets", 1);
        return true;
    }

    logWithInfo("Start describing");

    const auto& userDataTablePath = GetUserDataTablePath(YdbClient);
    const auto& idToCryptaIdTablePath = GetIdToCryptaIdTablePath(YdbClient);

    TMaybe<ui64> lastUserId = 0;
    ui64 processedUsersCount = 0;

    auto getCryptaIdsFuture = GetCryptaIdsForUsers(YdbClient, userSetId, *lastUserId, DescribingBatchSize, idToCryptaIdTablePath);
    DeleteOldStats(userSetId);

    while (lastUserId.Defined() && processedUsersCount < MaxUserProcessedCount) {
        TScopeTimer timer(Stats.Percentile, "describe_user_set.timing.batch_processing");
        const auto getCryptaIdsResult = WaitFor(getCryptaIdsFuture).ValueOrThrow();
        ThrowOnError(getCryptaIdsResult, AddMessagePrefix(userSetId, "Get crypta ids error"));

        const auto& userInfo = ParseUsersInfo(getCryptaIdsResult.GetResultSet(0));
        processedUsersCount += userInfo.ProcessedUsersCount;
        lastUserId = userInfo.LastUserId;
        if (lastUserId.Defined()) {
            getCryptaIdsFuture = GetCryptaIdsForUsers(YdbClient, userSetId, *lastUserId, DescribingBatchSize, idToCryptaIdTablePath);
        }

        DescribeCryptaIds(userSetId, userDataTablePath, getCryptaIdsResult.GetResultSet(1));

        Stats.Count->Add("describe_user_set.count.described_users", userInfo.ProcessedUsersCount);

        UpdateStatsIfNecessary(userSetId, processedUsersCount);
    }

    UpdateStats(userSetId, processedUsersCount, true);
    logWithInfo("Describing finished");
    return true;
}

TString TDescribeUserSetProcessor::AddMessagePrefix(TUserSetId userSetId, const TString& message) {
    return TStringBuilder() << "[TDescribeUserSetProcessor] [User set id = " << userSetId << "] " << message;
}
