#pragma once

#include "id_with_hash.h"

#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/identifiers/lib/id_types/all.h>
#include <crypta/lib/native/ydb/types.h>
#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>
#include <crypta/siberia/bin/common/data/id.h>
#include <crypta/siberia/bin/common/data/proto/user.pb.h>
#include <crypta/siberia/bin/common/describing/proto/describe_cmd.pb.h>
#include <crypta/siberia/bin/common/ydb/paths/paths.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <yt/yt/core/actions/future.h>

#include <util/generic/string.h>
#include <util/string/subst.h>

namespace NCrypta::NSiberia::NDescriber {
    struct TDescribeCryptaIdsDbRequest {
        struct TRequestParams {
            TString UserDataTablePath;
            const TVector<ui64> CryptaIds;
            size_t StartIndex = 0;
            size_t EndIndex = 0;
        };

        static constexpr const char* const Query = R"(
            PRAGMA TablePathPrefix("%s");
            PRAGMA SimpleColumns = '1';

            DECLARE $crypta_ids AS "List<Struct<
                crypta_id: Uint64
            >>";

            SELECT
                user_data_stats.*
            FROM AS_TABLE($crypta_ids) AS crypta_ids
            JOIN [{user_data_table}] AS user_data_stats
            ON crypta_ids.crypta_id == user_data_stats.crypta_id;
        )";

        static TString GetQuery(const TRequestParams& params) {
            TString query = Query;
            SubstGlobal(query, "{user_data_table}", params.UserDataTablePath);
            return query;
        }

        static NYdb::TParams GetParams(NYdb::TParamsBuilder&& paramsBuilder, const TRequestParams& params) {
            auto& idsParam = paramsBuilder.AddParam("$crypta_ids");

            idsParam.BeginList();
            for (size_t i = params.StartIndex; i < Min(params.EndIndex, params.CryptaIds.size()); ++i) {
                idsParam.AddListItem()
                    .BeginStruct()
                    .AddMember("crypta_id").Uint64(params.CryptaIds[i])
                    .EndStruct();
            }
            idsParam.EndList().Build();

            return paramsBuilder.Build();
        }
    };

    struct TGetCryptaIdsForIdsDbRequest {
        struct TRequestParams {
            TString IdToCryptaIdTablePath;
            const TVector<TIdWithHash>& Ids;
            size_t StartIndex = 0;
            size_t EndIndex = 0;
        };

        static constexpr const char* const Query = R"(
            PRAGMA TablePathPrefix("%s");
            PRAGMA SimpleColumns = '1';

            DECLARE $ids AS "List<Struct<
                hash: Uint64,
                type: String,
                id: String
            >>";

            SELECT
                crypta_id
            FROM AS_TABLE($ids) AS ids
            INNER JOIN [{id_to_crypta_id_table}] AS id_to_crypta_id
            ON ids.hash == id_to_crypta_id.id_hash AND ids.type == id_to_crypta_id.id_type AND ids.id == id_to_crypta_id.id;
        )";

        static TString GetQuery(const TRequestParams& params) {
            TString query = Query;
            SubstGlobal(query, "{id_to_crypta_id_table}", params.IdToCryptaIdTablePath);
            return query;
        }

        static NYdb::TParams GetParams(NYdb::TParamsBuilder&& paramsBuilder, const TRequestParams& params) {
            auto& idsParam = paramsBuilder.AddParam("$ids");

            idsParam.BeginList();
            for (size_t i = params.StartIndex; i < Min(params.EndIndex, params.Ids.size()); ++i) {
                const auto& item = params.Ids[i];
                idsParam.AddListItem()
                    .BeginStruct()
                    .AddMember("hash").Uint64(item.Hash)
                    .AddMember("type").String(item.Id.Type)
                    .AddMember("id").String(item.Id.Value)
                    .EndStruct();
            }
            idsParam.EndList().Build();

            return paramsBuilder.Build();
        }
    };

    struct TGetCryptaIdsForUsersDbRequest {
        struct TRequestParams {
            TString IdToCryptaIdTablePath;
            ui64 LastUserId = 0;
            ui64 BatchSize = 100;
        };

        static constexpr const char* const Query = R"(
            PRAGMA TablePathPrefix("%s");
            PRAGMA SimpleColumns = '1';

            DECLARE $lastUserId AS Uint64;
            DECLARE $batchSize AS Uint64;

            $userIds = (
                SELECT
                    id
                FROM {users_table}
                WHERE id > $lastUserId
                LIMIT $batchSize
            );

            $ids = (
                SELECT
                    attribute_value AS id,
                    attribute_key AS id_type,
                    Digest::CityHash(attribute_key || attribute_value) AS id_hash
                FROM $userIds AS user_ids
                JOIN {user_attributes_table} AS user_attributes
                ON user_attributes.user_id == user_ids.id
                WHERE String::StartsWith(user_attributes.attribute_key, "@")
            );

            $native_crypta_ids = (
                SELECT
                    CAST(id AS Uint64) AS crypta_id
                FROM $ids
                WHERE id_type == "@crypta_id" AND CAST(id AS Uint64) IS NOT NULL
            );

            $ordinary_ids = (
                SELECT
                    *
                FROM $ids
                WHERE id_type != "@crypta_id"
            );

            $crypta_ids_from_id_to_crypta_id = (
                SELECT
                    crypta_id
                FROM $ordinary_ids as ids
                JOIN [../{id_to_crypta_id_table}] AS id_to_crypta_id
                ON ids.id_hash == id_to_crypta_id.id_hash
                AND ids.id_type == id_to_crypta_id.id_type AND ids.id == id_to_crypta_id.id
            );

            SELECT
                MAX(id) AS last_user_id,
                COUNT(id) AS count
            FROM $userIds;

            SELECT * FROM $native_crypta_ids
            UNION ALL
            SELECT * FROM $crypta_ids_from_id_to_crypta_id;
        )";

        static TString GetQuery(const TRequestParams& params) {
            TString query = Query;
            SubstGlobal(query, "{users_table}", YDB_PATHS.GetUsersTable());
            SubstGlobal(query, "{user_attributes_table}", YDB_PATHS.GetUserAttributesTable());
            SubstGlobal(query, "{id_to_crypta_id_table}", params.IdToCryptaIdTablePath);
            return query;
        }

        static NYdb::TParams GetParams(NYdb::TParamsBuilder&& paramsBuilder, const TRequestParams& params) {
            return paramsBuilder
                .AddParam("$lastUserId").Uint64(params.LastUserId).Build()
                .AddParam("$batchSize").Uint64(params.BatchSize).Build()
                .Build();
        }
    };

    struct TDeleteStatsDbRequest {
        struct TRequestParams {
            ui64 SetId = 0;
        };

        static constexpr const char* const Query = R"(
            PRAGMA TablePathPrefix("%s");
            DECLARE $setId as Uint64;

            DELETE FROM {user_set_stats_table}
            WHERE user_set_id == $setId
        )";

        static TString GetQuery(const TRequestParams&) {
            TString query = Query;
            SubstGlobal(query, "{user_set_stats_table}", YDB_PATHS.GetUserSetStatsTable());
            return query;
        }

        static NYdb::TParams GetParams(NYdb::TParamsBuilder&& paramsBuilder, const TRequestParams& params) {
            return paramsBuilder
                .AddParam("$setId").Uint64(params.SetId).Build()
                .Build();
        }
    };

    NYdb::NTable::TAsyncDataQueryResult DescribeCryptaIds(TYdbClient& ydbClient, const TVector<ui64>& cryptaIds, const TString& userDataTablePath);
    NYdb::NTable::TAsyncDataQueryResult DescribeCryptaIds(TYdbClient& ydbClient, const TVector<ui64>& cryptaIds, size_t startIndex, size_t endIndex, const TString& userDataTablePath);
    NYdb::NTable::TAsyncDataQueryResult GetCryptaIdsForIds(TYdbClient& ydbClient, const TVector<TIdWithHash>& ids, size_t startIndex, size_t endIndex, const TString& idToCryptaIdTablePath);
    NYdb::NTable::TAsyncDataQueryResult GetCryptaIdsForUsers(TYdbClient& ydbClient, TUserSetId userSetId, TUserId lastUserId, ui64 batchSize, const TString& idToCryptaIdTablePath);
    NYdb::NTable::TAsyncDataQueryResult DeleteStats(TYdbClient& ydbClient, ui64 setId);
}
