import argparse
import logging
import os
import sys

import tvmauth

from crypta.lib.python import (
    time_utils,
    yaml_config,
)
from crypta.lib.python.logging import logging_helpers
import crypta.lib.python.tvm.helpers as tvm
from crypta.lib.python.ydb.ydb_client import YdbClient
from crypta.siberia.bin.common.data.proto import (
    user_set_status_pb2,
    user_set_type_pb2,
)
from crypta.siberia.bin.common.siberia_client import SiberiaClient
from crypta.siberia.bin.expirator.lib import (
    config_fields,
    ydb_request,
)


logger = logging.getLogger(__name__)

NOT_MATERIALIZED = user_set_type_pb2.TUserSetType().NotMaterialized
NOT_READY = user_set_status_pb2.TUserSetStatus().NotReady


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def make_request(siberia_client, row, tvm_ticket):
    if row.type == NOT_MATERIALIZED or row.status == NOT_READY:
        return siberia_client.user_sets_remove(row.id, tvm_ticket=tvm_ticket)
    else:
        return siberia_client.user_sets_remove_data(row.id, tvm_ticket=tvm_ticket)


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())
    logger.info("==================== Start ====================")

    args = parse_args()
    config = args.config
    logger.info(config)

    ydb_client = YdbClient(config[config_fields.YDB_ENDPOINT], config[config_fields.YDB_DATABASE], os.environ["YDB_TOKEN"])
    siberia_client = SiberiaClient(config[config_fields.SIBERIA_HOST], config[config_fields.SIBERIA_PORT])

    tvm_dst_id = config[config_fields.TVM_DST_ID]
    settings = tvmauth.TvmApiClientSettings(
        self_tvm_id=config[config_fields.TVM_SRC_ID],
        self_secret=os.environ[tvm.TVM_SECRET_ENV_VAR],
        dsts=[tvm_dst_id],
        localhost_port=tvm.get_tvm_test_port(),
    )

    tvm_client = tvmauth.TvmClient(settings)

    exit_code = 0
    current_time = time_utils.get_current_time()
    last_user_set_id = 0
    while True:
        rows_to_delete = ydb_request.get_expired_user_sets(ydb_client, current_time, last_user_set_id)

        if not rows_to_delete:
            break

        for row in rows_to_delete:
            try:
                make_request(siberia_client, row, tvm_client.get_service_ticket_for(tvm_id=tvm_dst_id))
            except Exception:
                logger.exception("Error while removing user set '%s'", row.id)
                exit_code = 1

            last_user_set_id = row.id

    if exit_code != 0:
        logger.error("Errors while removing user sets")

    logger.info("==================== Finish ====================")
    sys.exit(exit_code)
