#include <crypta/lib/native/cmd_args/parse_pb_options.h>
#include <crypta/lib/native/event_processing/worker.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/log/utils/log_exception.h>
#include <crypta/lib/native/log/wrappers/pqlib/pq_spdlog_wrapper.h>
#include <crypta/lib/native/pqlib/consumer.h>
#include <crypta/lib/native/pqlib/cookie_queue.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/reader.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/graphite/graphite_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/yaml/parse_yaml_file.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/native/ydb/ydb_client.h>
#include <crypta/siberia/bin/mutator/bin/cmd_args.pb.h>
#include <crypta/siberia/bin/mutator/lib/config/proto/config.pb.h>
#include <crypta/siberia/bin/mutator/lib/logic/processor_factory.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <yt/yt/core/actions/invoker.h>
#include <yt/yt/core/concurrency/thread_pool.h>
#include <yt/yt/core/misc/shutdown.h>

#include <util/datetime/base.h>
#include <util/stream/file.h>
#include <util/string/split.h>
#include <util/string/strip.h>

using namespace NCrypta;
using namespace NCrypta::NLog;
using namespace NCrypta::NPQ;
using namespace NCrypta::NSiberia;
using namespace NCrypta::NSiberia::NMutator;

namespace {
    TConfig ParseArgs(int argc, const char** argv) {
        const auto& args = ParsePbOptions<TCmdArgs>(argc, argv);
        const auto& yamlConfig = ParseYamlFile(args.GetConfigFile());

        return Yaml2Proto<TConfig>(yamlConfig);
    }

    THolder<TStatsWriter> GetStatsWriter(const TConfig& config) {
        Y_UNUSED(config);

        auto* stats = Singleton<TStatsRegistry>();
        auto statsWriter = MakeHolder<TStatsWriter>(*stats, TDuration::Minutes(1));
        if (config.HasGraphite()) {
            statsWriter->AddSink(MakeHolder<TGraphiteStatsSink>(config.GetGraphite()));
        }
        if (config.HasSolomon()) {
            statsWriter->AddSink(MakeHolder<TSolomonStatsSink>(config.GetSolomon()));
        }

        return statsWriter;
    }

    TConsumer CreateConsumer(const TLogbrokerConfig& config, TStats& pqlibStats) {
        auto pqlibLogger = MakeIntrusive<NCrypta::NLog::TPqSpdlogWrapper>(config.GetPqlibLogName());
        return TConsumer(
            NPQLib::Create(config.GetPqLib()),
            pqlibLogger,
            config.GetConsumer(),
            NCredentialsProvider::Create(config.GetCredentials(), pqlibLogger),
            pqlibStats
        );
    }

    TStats::TSettings GetStatsSettings() {
        return {
            .HistMax = 10,
            .HistBinCount = 10
        };
    }
}

int main(int argc, const char** argv) {
    TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

    try {
        const auto& config = ParseArgs(argc, argv);

        RegisterLogs(config.GetLogs());

        auto log = NLog::GetLog("main");

        auto statsWriter = GetStatsWriter(config);
        statsWriter->Start();

        const auto& statsSettings = GetStatsSettings();

        TStats pqlibStats("pqlib", statsSettings);
        TStats workerStats("worker", statsSettings);

        TYdbClient ydbClient(config.GetYdb());

        auto consumer = CreateConsumer(config.GetLogbroker(), pqlibStats);

        NPQ::TCookieQueue cookiesToCommit;

        TProcessorFactory processorFactory(config.GetProcessor(), ydbClient, workerStats);
        auto worker = NYT::New<NEventProcessing::TWorker>(processorFactory, cookiesToCommit, workerStats);

        auto threadPool = NYT::New<NYT::NConcurrency::TThreadPool>(config.GetWorker().GetThreads(), "worker");

        NPQ::TReader reader(
            TDuration::MilliSeconds(1000),
            consumer,
            cookiesToCommit,
            [&threadPool, &worker](auto&& readResult) {
                threadPool->GetInvoker()->Invoke(BIND([&worker, readResult = std::move(readResult)] () mutable {
                    worker->Process(std::move(readResult));
                }));
            },
            NLog::GetLog(config.GetLogbroker().GetPqlibLogName())
        );
        reader.SetOnFinishHandler([&stopSignalWaiter] () {
            stopSignalWaiter.Signal();
        });

        log->info("Start");
        reader.Start();

        stopSignalWaiter.Wait();

        log->info("Stop");
    } catch (const yexception& e) {
        LogException(e.what());
        return 1;
    } catch (const std::exception& e) {
        LogException(e.what());
        return 1;
    } catch (...) {
        LogException("Unknown exception");
        return 1;
    }

    NYT::Shutdown();

    return 0;
}
