#include "segmentate_command_processor.h"

#include <crypta/lib/native/concurrency/wait_for.h>
#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/ydb/helpers.h>
#include <crypta/lib/native/ydb/join.h>
#include <crypta/siberia/bin/common/data/proto/segment.pb.h>
#include <crypta/siberia/bin/common/data/proto/user.pb.h>
#include <crypta/siberia/bin/common/segmentation/engine/helpers.h>
#include <crypta/siberia/bin/common/segmentation/engine/rule_applier.h>
#include <crypta/siberia/bin/common/ydb/parse_utils/parse_utils.h>
#include <crypta/siberia/bin/common/ydb/paths/paths.h>
#include <crypta/siberia/bin/common/ydb/requests/add_users_to_segment_db_request.h>
#include <crypta/siberia/bin/common/ydb/requests/get_segment_db_request.h>
#include <crypta/siberia/bin/common/ydb/requests/update_segment_db_request.h>
#include <crypta/siberia/bin/common/ydb/requests/user_set_exists_db_request.h>

#include <library/cpp/yson/node/node_io.h>

using namespace NCrypta;
using namespace NCrypta::NSiberia;
using namespace NCrypta::NSiberia::NSegmentator;

namespace {
    void FillAttributes(NYT::TNode::TMapType& attributes, const TString& key, const TString& value) {
        const auto& it = attributes.find(key);

        if (it == attributes.end()) {
            attributes[key] = value;
        } else if (it->second.IsList()) {
            it->second.Add(value);
        } else {
            it->second = NYT::TNode::CreateList().Add(it->second).Add(value);
        }
    }
}

TSegmentateCommandProcessor::TSegmentateCommandProcessor(
    TSegmentateCommand command,
    TYdbClient& ydbClient,
    ::TStats& stats,
    ui64 batchSizeToWrite
)
    : TBase(ydbClient, stats)
    , Command(std::move(command))
    , BatchSizeToWrite(batchSizeToWrite)
{}

bool TSegmentateCommandProcessor::ProcessUnsafe() {
    TScopeTimer timer(Stats.Percentile, "timing.segmentate");
    Stats.Count->Add("command.count.segmentate");

    const auto& userSetId = Command.GetUserSetId();
    const auto& segmentId = Command.GetSegmentId();
    const auto& userSetPath = YdbClient.GetAbsolutePath(GetUserSetDirPath(userSetId));

    if (!IsUserSetExists(YdbClient, userSetId)) {
        Log->info("User set not found. Nothing to do");
        return true;
    }

    const auto& segment = GetSegment(YdbClient, userSetId, segmentId);
    if (!segment.Defined()) {
        Log->info("Segment not found. Nothing to do");
        return true;
    }

    const auto& parsingResult = NRuleParser::Parse(segment->GetRule());
    Y_ENSURE(parsingResult.Expression.Defined(), parsingResult.ErrorMessage);

    TRuleApplier ruleApplier(*parsingResult.Expression);

    size_t rowCount = 0;
    const auto& attributesTable = JoinYdbPath(userSetPath, "user_attributes");

    const auto createSessionResult = WaitFor(YdbClient.TableClient.GetSession()).ValueOrThrow();
    ThrowOnError(createSessionResult);
    auto session = createSessionResult.GetSession();

    TVector<TUserId> userIds;
    userIds.reserve(BatchSizeToWrite);
    TMaybe<ui64> lastRowUserId = Nothing();
    NYT::TNode::TMapType userAttributes;
    auto it = WaitFor(session.ReadTable(attributesTable)).ValueOrThrow();

    for (auto streamPart = WaitFor(it.ReadNext()).ValueOrThrow(); !streamPart.EOS(); streamPart = WaitFor(it.ReadNext()).ValueOrThrow()) {
        const auto& resultSet = streamPart.GetPart();
        NYdbCommonParseUtils::TUserAttributesResultSetParser parser(resultSet);
        for (; parser.TryNextRow(); ++rowCount) {
            const auto& row = parser.GetRow();

            if (!lastRowUserId.Defined()) {
                lastRowUserId = row.UserId;
            } else if (lastRowUserId != row.UserId) {
                if (ruleApplier.IsSatisfy(userAttributes)) {
                    userIds.push_back(*lastRowUserId);
                    if (userIds.size() >= BatchSizeToWrite) {
                        AddUsersToSegment(YdbClient, userSetId, segmentId, userIds);
                        userIds.clear();
                    }
                }
                userAttributes.clear();
                lastRowUserId = row.UserId;
            }

            FillAttributes(userAttributes, row.Key, row.Value);
        }
    }

    if (!userAttributes.empty() && ruleApplier.IsSatisfy(userAttributes)) {
        userIds.push_back(*lastRowUserId);
        userAttributes.clear();
    }

    if (!userIds.empty()) {
        AddUsersToSegment(YdbClient, userSetId, segmentId, userIds);
        userIds.clear();
    }

    UpdateSegment(YdbClient, userSetId, TSegmentUpdateInfo{.Id = segmentId, .Status = "ready"});

    return true;
}
