import argparse
import logging
import os
import sys

import requests
import tvmauth
import yt.wrapper as yt

from crypta.lib.python import yaml_config
from crypta.lib.python.logging import logging_helpers
import crypta.lib.python.tvm.helpers as tvm
from crypta.lib.python.yt.http_mapper import utils as yt_http_utils
from crypta.siberia.bin.common.data.proto.user_set_status_pb2 import TUserSetStatus
from crypta.siberia.bin.common.siberia_client import SiberiaClient
from crypta.siberia.bin.users_uploader.lib import (
    config_fields,
    upload_users_mapper,
)


TVM_SECRET_ENV_VAR = "TVM_SECRET"
logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def get_tvm_client(tvm_src_id, tvm_dst_id):
    settings = tvmauth.TvmApiClientSettings(
        self_tvm_id=tvm_src_id,
        self_secret=os.environ[tvm.TVM_SECRET_ENV_VAR],
        dsts=[tvm_dst_id],
        localhost_port=tvm.get_tvm_test_port(),
    )

    return tvmauth.TvmClient(settings)


def get_job_config(raw_job_config, tvm_src_id, tvm_dst_id, user_set_id):
    raw_job_config.update({
        config_fields.TVM_SRC_ID: tvm_src_id,
        config_fields.TVM_DST_ID: tvm_dst_id,
    })
    job_config = yt_http_utils.tvm_job_config_from_args(raw_job_config, [config_fields.MAX_RETRIES, config_fields.FIELDS_ID_TYPES])
    job_config.url_template = upload_users_mapper.URL_TEMPLATE_UPLOAD.format(user_set_id=user_set_id)
    return job_config


def send_table(src_table, hosts, job_config, tvm_secret):
    logger.info("Sending table %s", src_table)

    secure_vault_spec = {"secure_vault": {
        upload_users_mapper.UploadUsersMapper.TVM_SECRET_ENV_VAR: tvm_secret
    }}

    yt_http_utils.run_map(upload_users_mapper.UploadUsersMapper, hosts, job_config, src_table, additional_spec=secure_vault_spec)
    logger.info("Done.")


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())
    logger.info("================ Start ================")

    args = parse_args()
    config = args.config
    logger.info(config)

    yt.config.set_proxy(config[config_fields.YT_PROXY])
    yt.config["pool"] = config[config_fields.YT_POOL]

    src_table = config[config_fields.SOURCE_TABLE]
    logger.info("Src table: %s", src_table)

    tvm_src_id = config[config_fields.TVM_SRC_ID]
    tvm_dst_id = config[config_fields.TVM_DST_ID]

    siberia_host = config[config_fields.SIBERIA_HOST]
    siberia_port = config[config_fields.SIBERIA_PORT]

    user_set_id = config[config_fields.USER_SET_ID]

    tvm_client = get_tvm_client(tvm_src_id, tvm_dst_id)
    siberia_client = SiberiaClient(siberia_host, siberia_port)

    job_config = get_job_config(config[config_fields.JOB_CONFIG], tvm_src_id, tvm_dst_id, user_set_id)
    exit_code = 0

    try:
        tvm_ticket = tvm_client.get_service_ticket_for(tvm_id=tvm_dst_id)
        user_set = siberia_client.user_sets_get(user_set_id, tvm_ticket=tvm_ticket)

        if user_set.Status == TUserSetStatus().NotReady:
            send_table(src_table, ["{}:{}".format(siberia_host, siberia_port)], job_config, tvm_secret=os.environ[tvm.TVM_SECRET_ENV_VAR])
            siberia_client.user_sets_update(user_set_id, status=TUserSetStatus().Ready, tvm_ticket=tvm_ticket)

        siberia_client.user_sets_describe(user_set_id, tvm_ticket=tvm_ticket)
    except requests.HTTPError as e:
        if e.response.status_code != requests.codes.not_found:
            logger.exception("Got HTTP error")
            exit_code = 1
        else:
            logger.info("User set not found. Nothing to do")
    except Exception:
        logger.exception("Failed while uploading %s", src_table)
        exit_code = 1

    logger.info("================ Finish ================")
    sys.exit(exit_code)
