import collections

from library.python.protobuf.json import proto2json

from crypta.lib.python.identifiers import identifiers
from crypta.lib.python.yt.http_mapper import tvm_http_requests_mapper
from crypta.siberia.bin.common.data.proto import user_pb2
from crypta.siberia.bin.core.proto import add_users_request_pb2


URL_TEMPLATE_UPLOAD = "/users/add?user_set_id={user_set_id}"


def get_siberia_id_type(id_type):
    return "@{}".format(id_type)


def is_attr_key_correct(key):
    return not key.startswith("@")


def is_utf8(s):
    try:
        s.decode("utf-8")
        return True
    except UnicodeDecodeError:
        return False


def serialize_attributes(row, fields_id_types):
    res = {}

    for key, value in row.iteritems():
        if not value or not is_attr_key_correct(key):
            continue

        str_value = str(value)
        str_value = str_value if is_utf8(str_value) else ""  # Quick fix https://st.yandex-team.ru/CRYPTAYT-3642
        id_type = fields_id_types.get(key)

        if id_type is not None:
            id = identifiers.GenericID(id_type, str_value)
            values = [id.normalize if id.is_valid() else str_value]
            res[get_siberia_id_type(id_type)] = user_pb2.TUser.TInfo.TAttributeValues(Values=values)
        else:
            values = [str_value]
            res[key] = user_pb2.TUser.TInfo.TAttributeValues(Values=values)
    return res


def serialize_users(batch, fields_id_types):
    return proto2json.proto2json(add_users_request_pb2.TAddUsersRequest(
        Users=[user_pb2.TUser.TInfo(Status="active", Attributes=serialize_attributes(row, fields_id_types)) for row in batch]
    ), proto2json.Proto2JsonConfig(map_as_object=True))


def get_hashable_batch(batch):
    return frozenset((frozenset([str(x) for x in row.items()]) for row in batch))


class UploadUsersMapper(tvm_http_requests_mapper.TvmHttpRequestsMapper):
    def __init__(self, max_retries, fields_id_types=None, *args, **kwargs):
        self.retry_counter = collections.Counter()
        self.max_retries = max_retries
        self.fields_id_types = fields_id_types or {}

        super(UploadUsersMapper, self).__init__(*args, **kwargs)

    def is_request_retriable(self, request):
        return (self.retry_counter[get_hashable_batch(request.batch)] < self.max_retries) and \
               (request.status == "exception" or not (400 <= int(request.status) < 500))

    def get_request_wo_headers(self, batch):
        yield "POST", self.url, serialize_users(batch, self.fields_id_types)

    def retry_failed_request(self, request):
        self.retry_counter[get_hashable_batch(request.batch)] += 1
        return self.is_request_retriable(request)

    def process_response(self, request):
        batch = request.batch
        hashable_batch = get_hashable_batch(batch)
        request_succeded = request.ok or request.status == "404"

        if not request_succeded and not self.is_request_retriable(request):
            raise Exception("Upload error. Batch: {}\nStatus:\n{}\nData:\n{}".format(batch, request.status, request.response_body))
        elif request_succeded and hashable_batch in self.retry_counter:
            del self.retry_counter[hashable_batch]

        return []

    def get_headers(self):
        headers = super(UploadUsersMapper, self).get_headers()
        headers["Content-Type"] = "application/json"
        return headers
