# encoding: utf-8

import collections
from datetime import timedelta as dt

from yabs.server.proto.keywords import keywords_data_pb2

from crypta.lib.python import crypta_env
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import (
    arcadia_ci,
    deploy,
    event_processing_utils,
)
from crypta.lib.python.spine.consts import (
    environment,
    dc as dc_consts,
)
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler.aggregate_hosts import AggregateHosts
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.juggler.juggler_check_generator import JugglerCheckGenerator
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.solomon import solomon_check_generator
from crypta.lib.python.spine.yasm import yasm_dashboard_generator
from sandbox.projects.crypta import merge_to_bigb_collector
import sandbox.projects.crypta.common.task as crypta_common_task
from sandbox.projects.crypta.siberia import (
    expirator,
    make_crypta_id_user_data,
    make_user_data_encoded,
    matching_table_uploader,
    user_data_uploader,
)

COMMON_AGGR_HOSTS = AggregateHosts(testing_host="crypta_siberia_testing", production_host="crypta_siberia")
CRYPTA_SIBERIA_NETS = "_CRYPTA_SIBERIA_NETS_"
PROJECT = "crypta_siberia"
DESCRIBER = "describer"
MUTATOR = "mutator"
SEGMENTATOR = "segmentator"
CORE = "core"
CUSTOM_AUDIENCE_SUGGESTER = "custom_audience_suggester"
CONSUMER = "crypta/prod/siberia/consumer"
Task = collections.namedtuple("Task", ["task", "aggr_hosts", "yt_deps", "schedule_interval", "kill_timeout", "crit_time"])
UniversalTask = collections.namedtuple("UniversalTask", ["bundle_name", "cmd", "schedule_interval", "kill_timeout", "crit_time", "aggr_hosts", "yt_deps"])


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-siberia"])
    yasm = juggler.add_subregistry(yasm_dashboard_generator.CryptaYtYasmDashboardGenerator("Siberia"))

    ca_release = arcadia_ci.CryptaReleaseGenerator(juggler.clone(host="crypta-custom-audience")).standard_release(
        title="Custom Audience",
        abs_paths=["crypta/siberia/bin/custom_audience/**"],
    )

    add_sandbox_tasks(juggler)
    add_describer(juggler, yasm)
    add_mutator(juggler, yasm)
    add_core(juggler, yasm)
    add_segmentator(juggler, yasm)

    add_custom_audience_sandbox_tasks(juggler, ca_release)
    add_custom_audience_service(juggler, ca_release)
    add_custom_audience_suggester_service(juggler, ca_release)

    return juggler


def add_sandbox_tasks(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["SIBERIA"])

    for task in [
        Task(
            task=expirator.CryptaSiberiaExpiratorTask,
            aggr_hosts=COMMON_AGGR_HOSTS,
            yt_deps=[],
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=1),
            crit_time=dt(days=2),
        ),
        Task(
            task=matching_table_uploader.CryptaSiberiaMatchingTableUploaderTask,
            aggr_hosts=COMMON_AGGR_HOSTS,
            yt_deps=YtProxy.Group.offline,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=22),
            crit_time=dt(days=2),
        ),
        Task(
            task=user_data_uploader.CryptaSiberiaUserDataUploaderTask,
            aggr_hosts=COMMON_AGGR_HOSTS,
            yt_deps=YtProxy.Group.offline,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=18),
            crit_time=dt(days=2),
        ),
        Task(
            task=make_crypta_id_user_data.CryptaSiberiaMakeCryptaIdUserDataTask,
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            yt_deps=YtProxy.Group.offline,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=8),
            crit_time=dt(days=2),
        ),
        Task(
            task=make_user_data_encoded.CryptaSiberiaMakeUserDataEncodedTask,
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            yt_deps=YtProxy.Group.offline,
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=3),
            crit_time=dt(days=1),
        ),
    ]:
        for aggr_host in task.aggr_hosts:
            sandbox.create_scheduler(
                task.task,
                schedule_interval=task.schedule_interval,
                kill_timeout=task.kill_timeout,
                env=aggr_host.environment,
            ).check(task.crit_time).set_host(aggr_host.host).add_yt_dependencies(task.yt_deps)

    for task in [
        UniversalTask(
            bundle_name="siberia-make-id-to-crypta-id",
            cmd=["{{cwd}}/crypta-siberia-make-id-to-crypta-id", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(hours=12),
            kill_timeout=dt(hours=12),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            yt_deps=YtProxy.Group.offline,
        ),
    ]:
        for aggr_host in task.aggr_hosts:
            sandbox.create_run_universal_bundle_scheduler(
                bundle_name=task.bundle_name,
                cmd=task.cmd,
                env=aggr_host.environment,
                schedule_interval=task.schedule_interval,
                kill_timeout=task.kill_timeout,
            ).check(crit_time=task.crit_time).set_host(aggr_host.host).add_yt_dependencies(task.yt_deps)

    bundle_name = "siberia-convert-to-user-data-stats"
    for input_table_postfix, output_table_postfix, juggler_service_postfix in (
        ("lab/data/crypta_id/UserData", "siberia/custom/crypta_id_user_data/by_crypta_id", 'crypta-id'),
        ("lab/data/all_crypta_id/UserData", "siberia/custom/crypta_id_user_data/by_all_crypta_id", "all-crypta-id"),
    ):
        sandbox.create_run_universal_bundle_scheduler(
            bundle_name=bundle_name,
            cmd=["{{cwd}}/crypta-siberia-convert-to-user-data-stats", "--config", "{{cwd}}/config.yaml"],
            env=COMMON_AGGR_HOSTS.production_host.environment,
            schedule_interval=dt(hours=12),
            kill_timeout=dt(hours=12),
            template_rendering_context={
                "input_table_postfix": input_table_postfix,
                "output_table_postfix": output_table_postfix,
            },
            juggler_service="{}_{}".format(bundle_name, juggler_service_postfix),
        ).check(crit_time=dt(days=2)).set_host(COMMON_AGGR_HOSTS.production_host.host).add_yt_dependencies(YtProxy.Group.offline)

    _add_custom_user_data_upload(sandbox)


def add_custom_audience_sandbox_tasks(juggler, ca_release):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["CUSTOM_AUDIENCE"])

    for task in [
        UniversalTask(
            bundle_name=ca_release.add_universal_bundle("crypta/siberia/bin/custom_audience/ca_builder/bundle/crypta-siberia-custom-audience-builder.json").bundle_id,
            cmd=["{{cwd}}/crypta-siberia-custom-audience-builder", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(hours=12),
            kill_timeout=dt(hours=12),
            crit_time=dt(days=2),
            aggr_hosts=COMMON_AGGR_HOSTS,
            yt_deps=YtProxy.Group.offline,
        ),
        UniversalTask(
            bundle_name=ca_release.add_universal_bundle("crypta/siberia/bin/custom_audience/hosts_filter/bundle/crypta-siberia-custom-audience-hosts-filter.json").bundle_id,
            cmd=["{{cwd}}/crypta-siberia-custom-audience-hosts-filter", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=2),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            yt_deps=YtProxy.Group.offline,
        ),
        UniversalTask(
            bundle_name=ca_release.add_universal_bundle("crypta/siberia/bin/custom_audience/bs_host_cluster_mapping/bundle/crypta-siberia-custom-audience-bs-host-cluster-mapping.json").bundle_id,
            cmd=["{{cwd}}/crypta-siberia-custom-audience-bs-host-cluster-mapping", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=2),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            yt_deps=YtProxy.Group.offline,
        ),
        UniversalTask(
            bundle_name=ca_release.add_universal_bundle("crypta/siberia/bin/custom_audience/build_apps_for_suggester/bundle/crypta-siberia-custom-audience-build-apps-for-suggester.json").bundle_id,
            cmd=["{{cwd}}/crypta-siberia-custom-audience-build-apps-for-suggester", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=2),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            yt_deps=YtProxy.Group.offline,
        ),
        UniversalTask(
            bundle_name=ca_release.add_universal_bundle("crypta/siberia/bin/custom_audience/bs_apps_cluster_mapping/bundle/crypta-siberia-custom-audience-bs-apps-cluster-mapping.json").bundle_id,
            cmd=["{{cwd}}/crypta-siberia-custom-audience-bs-apps-cluster-mapping", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=2),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host],
            yt_deps=YtProxy.Group.offline,
        ),
    ]:
        for aggr_host in task.aggr_hosts:
            sandbox.create_run_universal_bundle_scheduler(
                bundle_name=task.bundle_name,
                cmd=task.cmd,
                env=aggr_host.environment,
                schedule_interval=task.schedule_interval,
                kill_timeout=task.kill_timeout,
            ).check(crit_time=task.crit_time).set_host(aggr_host.host).add_yt_dependencies(task.yt_deps)

    bundle_name = ca_release.add_universal_bundle("crypta/siberia/bin/custom_audience/to_bigb_collector/bundle/crypta-siberia-custom-audience-to-bigb-collector.json").bundle_id
    for source_dir, table_suffix, id_type, keyword_id in (
        ("//home/crypta/production/siberia/custom_audience/sites_clustering/cryptaid_clusterid", "crypta_id_hosts", "crypta_id", keywords_data_pb2.KW_CRYPTA_CUSTOM_AUDIENCE_HOST_CLUSTERS),
        ("//home/crypta/production/siberia/custom_audience/sites_clustering/yandexuid_clusterid", "yandexuid_hosts", "yandex_id", keywords_data_pb2.KW_CRYPTA_CUSTOM_AUDIENCE_HOST_CLUSTERS),
    ):
        sandbox.create_run_universal_bundle_scheduler(
            bundle_name=bundle_name,
            cmd=["{{cwd}}/crypta-siberia-custom-audience-to-bigb-collector", "--config", "{{cwd}}/config.yaml"],
            env=COMMON_AGGR_HOSTS.production_host.environment,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=2),
            template_rendering_context={
                "source_dir": source_dir,
                "table_suffix": table_suffix,
                "id_type": id_type,
                "keyword_id": keyword_id,
            },
            juggler_service="{}_{}".format(bundle_name, table_suffix)
        ).check(crit_time=dt(days=2)).set_host(COMMON_AGGR_HOSTS.production_host.host).add_yt_dependencies(YtProxy.Group.offline)

    suffix = "ca-export"
    for aggr_host in COMMON_AGGR_HOSTS:
        directory = environment.PRODUCTION if aggr_host.environment == environment.STABLE else environment.TESTING
        sandbox.create_scheduler(
            merge_to_bigb_collector.CryptaMergeToBigbCollectorTask,
            kill_timeout=dt(hours=1),
            schedule_interval=dt(minutes=15),
            env=aggr_host.environment,
            extra_params={
                "pool": "crypta_cdp",
                "fresh_dir": "//home/crypta/{}/siberia/custom_audience/collectors".format(directory),
                "output_dir": "//home/crypta/{}/siberia/custom_audience/to_bigb".format(directory),
                "juggler_service_suffix": suffix,
            },
        ).check(
            crit_time=dt(hours=2),
            juggler_service=merge_to_bigb_collector.CryptaMergeToBigbCollectorTask.get_juggler_service_with_suffix(suffix),
        ).set_host(aggr_host.host).add_yt_dependencies(YtProxy.Group.offline)


def add_describer(juggler_check_generator, yasm):
    yasm.add_devops(service="Describer", prj_tag="crypta-siberia-describer")

    for dc, env, nanny_service in [
        ("sas", environment.TESTING, "test_crypta_sib_describer_sas"),
        ("sas", environment.TESTING, "test_crypta_sib_describer_s_sas"),
        ("sas", environment.PRODUCTION, "prod_crypta_sib_describer_sas"),
        ("sas", environment.PRODUCTION, "prod_crypta_sib_describer_s_sas"),
        ("iva", environment.PRODUCTION, "prod_crypta_sib_describer_iva"),
        ("iva", environment.PRODUCTION, "prod_crypta_sib_describer_s_iva"),
    ]:
        juggler = juggler_check_generator.clone(**_get_juggler_config(nanny_service).__dict__)

        juggler.icmpping().set_crit_limit(1).set_phone_escalation(env == environment.PRODUCTION)

        solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=DESCRIBER,
            cluster=env,
            dc=dc,
        )

        event_processing_utils.get_invalid_events_check(solomon)
        event_processing_utils.get_unsupported_events_check(solomon)
        event_processing_utils.get_zero_threshold_check(solomon.get_sensor("worker.describe_ids.batch.errors.empty_batch"), "empty batches: {{ pointValue }}")
        event_processing_utils.get_zero_threshold_check(solomon.get_sensor("worker.describe_ids.batch.errors.no_valid_id"), "batches without valid ids: {{ pointValue }}")
        event_processing_utils.get_zero_threshold_check(solomon.get_sensor("worker.describe_ids.count.ids_with_unknown_type"), "unknown id types: {{ pointValue }}")
        event_processing_utils.get_zero_threshold_check(solomon.get_sensor("worker.errors.unknown"), "unknown errors: {{ pointValue }}")

        solomon.get_sensor("worker.describe_ids.count.ids_with_invalid_value").create_threshold_check(
            aggregation=alert_pb2.SUM,
            predicate=alert_pb2.GT,
            threshold=100000,
            period=dt(minutes=10),
            description="invalid ids: {{ pointValue }}",
        ).add_nodata_mode(consts.NoDataMode.force_ok)

        solomon.get_sensor("worker.errors.ydb").create_threshold_check(
            aggregation=alert_pb2.SUM,
            predicate=alert_pb2.GT,
            threshold=300,
            period=dt(hours=1),
            description="YDB errors: {{ pointValue }}",
        ).add_nodata_mode(consts.NoDataMode.force_ok)

    _add_lag_check(juggler_check_generator, DESCRIBER, "describe-log", 50, True)
    _add_lag_check(juggler_check_generator, DESCRIBER, "describe-slow-log", 10000, False)


def add_mutator(juggler_check_generator, yasm):
    yasm.add_devops(service="Mutator", prj_tag="crypta-siberia-mutator")

    for dc, env, nanny_service in [
        ("sas", environment.TESTING, "test_crypta_sib_mutator_sas"),
        ("sas", environment.PRODUCTION, "prod_crypta_sib_mutator_sas"),
        ("iva", environment.PRODUCTION, "prod_crypta_sib_mutator_iva"),
    ]:
        juggler = juggler_check_generator.clone(**_get_juggler_config(nanny_service).__dict__)
        juggler.icmpping().set_crit_limit(1)

        solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=MUTATOR,
            cluster=env,
            dc=dc,
        )

        event_processing_utils.get_invalid_events_check(solomon),
        event_processing_utils.get_unsupported_events_check(solomon),
        event_processing_utils.get_zero_threshold_check(solomon.get_sensor("worker.error.remove_segment_data.exists"), "error count: {{ pointValue }}")
        event_processing_utils.get_zero_threshold_check(solomon.get_sensor("worker.error.remove_user_set_data.exists"), "error count: {{ pointValue }}")

    _add_lag_check(juggler_check_generator, MUTATOR, "change-log", 1000, False)


def add_core(juggler_check_generator, yasm):
    yasm.add_devops(service="Core", prj_tag="crypta-siberia-core")

    for dc, env, nanny_service in [
        ("sas", environment.TESTING, "test_crypta_sib_core_sas"),
        ("sas", environment.PRODUCTION, "prod_crypta_sib_core_sas"),
        ("iva", environment.PRODUCTION, "prod_crypta_sib_core_iva"),
    ]:
        config = _get_juggler_config(nanny_service)
        config.crit_limit = 1
        config.escalation = (env == environment.PRODUCTION)

        flap_detector_params = FlapDetectorParams(dt(minutes=5), dt(minutes=10))

        juggler = juggler_check_generator.clone(**config.__dict__)
        juggler.icmpping().add_flap_detector(flap_detector_params)
        juggler.http(
            "alive",
            path="/ping?client=monitoring",
            port=80,
        ).add_flap_detector(flap_detector_params)
        juggler.tvm_client_status()

        solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=CORE,
            cluster=env,
            dc=dc,
        )

        _get_timings_check(solomon, "describe_ids", dt(seconds=3), flap_detector_params, config.escalation),
        _get_timings_check(solomon, "get_user_set_stats", dt(seconds=20), flap_detector_params, config.escalation),
        _get_timings_check(solomon, "describe_user_set", dt(milliseconds=200), flap_detector_params, False),
        _get_timings_check(solomon, "add_users", dt(seconds=2), flap_detector_params, False),

        solomon.create_sensor_aggregation_check_by_host(
            service="5XX_codes",
            period=dt(minutes=2),
            sensor_glob="*.request.http_code.5??",
            predicate=alert_pb2.GT,
            threshold=100,
            description="Total 5XX codes : {{expression.total}} > {{expression.threshold}}",
        ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)


def add_segmentator(juggler_check_generator, yasm):
    yasm.add_devops(service="Segmentator", prj_tag="crypta-siberia-segmentator")

    for dc, env, nanny_service in [
        ("sas", environment.TESTING, "test_crypta_sib_segmentator_sas"),
        ("sas", environment.PRODUCTION, "prod_crypta_sib_segmentator_sas"),
        ("iva", environment.PRODUCTION, "prod_crypta_sib_segmentator_iva"),
    ]:
        juggler = juggler_check_generator.clone(**_get_juggler_config(nanny_service).__dict__)
        juggler.icmpping().set_crit_limit(1)

        solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=SEGMENTATOR,
            cluster=env,
            dc=dc,
        )
        event_processing_utils.get_invalid_events_check(solomon)
        event_processing_utils.get_unsupported_events_check(solomon)

    _add_lag_check(juggler_check_generator, SEGMENTATOR, "segmentate-log", 1000, False)


def add_custom_audience_service(base_juggler_check_generator, ca_release):
    grpc_port = 80
    http_port = 81

    ca_release.add_docker_image("crypta/siberia/bin/custom_audience/fast/docker/crypta-siberia-custom-audience.json")

    deploy_gen = base_juggler_check_generator.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage = deploy_gen.stage("crypta-siberia-custom-audience")
    deploy_unit = stage.multi_cluster_deploy_unit(
        pods_per_cluster={dc_consts.IVA: 2, dc_consts.MAN: 2, dc_consts.SAS: 2},
        cpu_ms=24000,
        ram=DataSize(gb=200),
        storage=deploy.SSD(
            capacity=DataSize(gb=256),
            bandwidth_guarantee=DataSize(mb=400),
            bandwidth_limit=DataSize(mb=400),
        ),
        project_network=CRYPTA_SIBERIA_NETS,
        network_bandwidth=DataSize(mb=10),
        id_=environment.PRODUCTION,
    ).add_endpoint_set(
        port=grpc_port,
    ).add_endpoint_set(
        port=http_port,
        id_="http",
    ).add_balancer(
        "ca.siberia.crypta.yandex.net"
    ).deployment_strategy(
        max_unavailable=1,
    )

    box = deploy_unit.box(
        docker_image="crypta/crypta-siberia-custom-audience",
    ).docker_release_rule(
        environment.STABLE,
    ).add_init_command(
        deploy.Command(
            "/root/init.sh",
        ).max_execution_time(
            dt(hours=2),
        ),
    ).literal_env(
        crypta_env.Production.crypta_environment
    ).literal_env(
        crypta_env.LiteralEnv("GRPC_PORT", str(grpc_port))
    ).literal_env(
        crypta_env.LiteralEnv("HTTP_PORT", str(http_port))
    ).secret_env(
        crypta_env.solomon_token
    ).secret_env(
        crypta_env.Production.yt_token
    )

    box.sandbox_release_rule(
        make_user_data_encoded.CryptaSiberiaMakeUserDataEncodedTask,
        environment.STABLE,
    ).static_resource(
        make_user_data_encoded.CryptaSiberiaEncodedUserDataInfo,
        "encoded_user_data_release_info",
        "/root/encoded_user_data_release_info.json",
    )

    box.workload(
        start_cmd="/root/service.sh",
        id_="grpc",
    ).liveness_cmd_check(
        deploy.Command(
            "/root/grpcurl -plaintext localhost:{} NCrypta.NSiberia.NCustomAudience.TCustomAudienceService.Ping".format(grpc_port)
        ),
    ).readiness_cmd_check(
        deploy.Command(
            "/root/grpcurl -plaintext localhost:{} NCrypta.NSiberia.NCustomAudience.TCustomAudienceService.Ready".format(grpc_port)
        ),
    )

    box.workload(
        start_cmd="/root/http_proxy.sh",
        id_="http_proxy",
    ).liveness_http_check(
        path="/ping",
        port=http_port,
    ).readiness_http_check(
        path="/ready",
        port=http_port,
    )

    juggler = deploy_unit.get_juggler_generator(base_juggler_check_generator.clone(
        warn_limit=0,
        crit_limit=100,
        escalation=False
    ))

    flap_detector_params = FlapDetectorParams(dt(minutes=5), dt(minutes=10))
    juggler.icmpping().add_flap_detector(flap_detector_params)
    juggler.http(
        "alive",
        path="/ping",
        port=http_port,
    ).add_flap_detector(flap_detector_params)

    juggler.http(
        "ready",
        path="/ready",
        port=http_port,
    ).add_flap_detector(flap_detector_params)


def add_custom_audience_suggester_service(base_juggler_check_generator, ca_release):
    grpc_port = 80
    http_port = 81

    ca_release.add_docker_image("crypta/siberia/bin/custom_audience/suggester/docker/crypta-siberia-custom-audience-suggester.json")

    deploy_gen = base_juggler_check_generator.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage = deploy_gen.stage("crypta-siberia-custom-audience-suggester")

    pods_per_env = {
        environment.TESTING: {dc_consts.IVA: 1, dc_consts.SAS: 1},
        environment.PRODUCTION: {dc_consts.IVA: 2, dc_consts.SAS: 2},
    }

    for env in environment.ALL:
        secrets = crypta_env.VARS_BY_ENV[env]

        deploy_unit = stage.multi_cluster_deploy_unit(
            pods_per_cluster=pods_per_env[env],
            cpu_ms=1000,
            ram=DataSize(gb=4),
            storage=deploy.HDD(
                capacity=DataSize(gb=4),
                bandwidth_guarantee=DataSize(mb=1),
                bandwidth_limit=DataSize(mb=1),
            ),
            project_network=CRYPTA_SIBERIA_NETS,
            network_bandwidth=DataSize(mb=1),
            id_=env,
        ).add_endpoint_set(
            port=grpc_port,
        ).add_endpoint_set(
            port=http_port,
            id_="http",
        ).add_balancer(
            "ca-suggester.siberia.crypta.yandex.net"
        ).deployment_strategy(
            max_unavailable=1,
        )

        box = deploy_unit.box(
            docker_image="crypta/crypta-siberia-custom-audience-suggester",
        ).docker_release_rule(
            environment.CRYPTA_ENV_TO_SANDBOX_ENV[env],
        ).add_init_command(
            deploy.Command(
                "/root/init.sh"
            ),
        ).literal_env(
            secrets.crypta_environment
        ).literal_env(
            crypta_env.LiteralEnv("GRPC_PORT", str(grpc_port))
        ).literal_env(
            crypta_env.LiteralEnv("HTTP_PORT", str(http_port))
        ).secret_env(
            crypta_env.solomon_token
        ).secret_env(
            secrets.yt_token
        )

        box.workload(
            start_cmd="/root/service.sh",
            id_="grpc",
        ).liveness_cmd_check(
            deploy.Command(
                "/root/grpcurl -plaintext localhost:{} NCrypta.NSiberia.NCustomAudience.NSuggester.TSuggesterService.Ping".format(grpc_port)
            ),
        ).readiness_cmd_check(
            deploy.Command(
                "/root/grpcurl -plaintext localhost:{} NCrypta.NSiberia.NCustomAudience.NSuggester.TSuggesterService.Ready".format(grpc_port)
            ),
        )

        box.workload(
            start_cmd="/root/http_proxy.sh",
            id_="http_proxy",
        ).liveness_http_check(
            path="/ping",
            port=http_port,
        ).readiness_http_check(
            path="/ready",
            port=http_port,
        )

        juggler = deploy_unit.get_juggler_generator(base_juggler_check_generator.clone(
            warn_limit=0,
            crit_limit=1,
            escalation=True,
        ))

        flap_detector_params = FlapDetectorParams(dt(minutes=10), dt(minutes=30))
        juggler.icmpping().add_flap_detector(flap_detector_params)
        juggler.http(
            "alive",
            path="/ping",
            port=http_port,
        ).add_flap_detector(flap_detector_params)

        juggler.http(
            "ready",
            path="/ready",
            port=http_port,
        ).add_flap_detector(flap_detector_params)

        solomon = solomon_check_generator.ServiceSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=CUSTOM_AUDIENCE_SUGGESTER,
            cluster=env,
        )

        for handle, threshold in (
            ("Suggest", 300 * 1000),
            ("GetHosts", 100 * 1000),
        ):
            solomon.get_sensor("grpc.{}.latency.wall_time.p95".format(handle)).create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.GT,
                threshold=threshold,
                period=dt(minutes=3),
                description="latency (μs): {{ pointValue }}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

            solomon.create_expression_alert_check_by_host(
                "{}.not.ok".format(handle),
                period=dt(hours=1),
                program="""
                let total = group_lines("sum",
                    group_by_time(1m, "sum", {labels_all})
                );
                let ok = group_lines("sum",
                    group_by_time(1m, "sum", {labels_ok})
                );
                let not_ok = sum(total - ok);
                let threshold = 0;

                alarm_if(not_ok > threshold);
                """.format(
                    labels_all=solomon.get_serialized_labels(_create_status_code_sensor(handle, "*")),
                    labels_ok=solomon.get_serialized_labels(_create_status_code_sensor(handle, "OK")),
                ),
                description="not ok codes: {{ expression.not_ok }}",
            ).add_nodata_mode(consts.NoDataMode.force_ok)

        for db, count in (
            ("segments", 100),
            ("hosts", 10**6),
        ):
            solomon.get_sensor("database.update.{}.timing".format(db)).create_threshold_check(
                aggregation=alert_pb2.LAST_NON_NAN,
                predicate=alert_pb2.GT,
                threshold=5 * 60 * 10**6,
                period=dt(hours=12),
                description="update time (μs): {{ pointValue }}",
            ).add_flap_detector(flap_detector_params)

            solomon.get_sensor("database.count.{}".format(db)).create_threshold_check(
                aggregation=alert_pb2.LAST_NON_NAN,
                predicate=alert_pb2.LT,
                threshold=count,
                period=dt(hours=12),
                description="count: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params)

        solomon.get_sensor("database.update").create_threshold_check(
            aggregation=alert_pb2.LAST_NON_NAN,
            predicate=alert_pb2.LTE,
            threshold=0,
            period=dt(hours=12),
            description="No updates for past 12 hours",
        ).add_flap_detector(flap_detector_params)


def _create_status_code_sensor(handle, code):
    return {
        "project": PROJECT,
        "cluster": environment.PRODUCTION,
        "service": CUSTOM_AUDIENCE_SUGGESTER,
        "sensor": "grpc.{}.request.status_code.{}".format(handle, code)
    }


def _get_juggler_config(nanny_service):
    return JugglerCheckGenerator.Config(
        child_group=nanny_service,
        child_group_type=consts.GroupType.nanny,
        escalation=False,
        warn_limit=0,
        crit_limit=0,
    )


def _get_timings_check(solomon, processor, threshold, flap_detector_params, escalation):
    return solomon.get_sensor("processors.{}.latency.wall_time.p95".format(processor)).create_threshold_check(
        aggregation=alert_pb2.MAX,
        predicate=alert_pb2.GT,
        threshold=threshold.total_seconds() * 10**6,
        period=dt(minutes=2),
        description="latency (μs): {{ pointValue }}",
    ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok).set_phone_escalation(escalation)


def _add_lag_check(juggler_check_generator, service, logname, threshold, escalation):
    return solomon_check_generator.ServiceSolomonCheckGenerator(
        juggler_check_generator,
        project=PROJECT,
        service=service,
        cluster=environment.PRODUCTION,
    ).create_logbroker_lag_check(
        consumer=CONSUMER,
        topic="crypta/prod/siberia/{}".format(logname),
        threshold=threshold,
        period=dt(minutes=5),
    ).add_flap_detector(FlapDetectorParams(dt(minutes=15), dt(minutes=30))).set_phone_escalation(escalation)


def _add_custom_user_data_upload(sandbox):
    task = user_data_uploader.CryptaSiberiaUserDataUploaderTask
    task_name = task.__name__
    custom_version = "by_crypta_id"
    juggler_service = "{}_{}".format(task_name, custom_version)

    for aggr_host in COMMON_AGGR_HOSTS:
        sandbox.create_scheduler(
            task,
            schedule_interval=dt(days=1),
            kill_timeout=dt(hours=18),
            env=aggr_host.environment,
            extra_params={
                "override_basic_params": True,
                "overridden_semaphore": "{}_{}_{}".format(task_name, aggr_host.environment, custom_version),
                "override_send_juggler_events": True,
                "override_juggler_host": crypta_common_task.JUGGLER_HOSTS[aggr_host.environment],
                "override_juggler_service": juggler_service,
                "custom_version": custom_version,
            },
        ).check(
            crit_time=dt(days=2),
            juggler_service=juggler_service,
        ).set_host(aggr_host.host).add_yt_dependencies(YtProxy.Group.offline)
