import collections
import os

import yt.wrapper as yt


class Node(object):
    def __init__(self, path, creation_time, type="table", attributes=dict()):
        for attr, value in attributes.items():
            self.__dict__[attr] = value

        self.path = path
        self.creation_time = creation_time
        self.type = type
        self.children = {}

    def get(self, attr):
        return self.__dict__[attr]

    def add(self, node):
        self.children[os.path.basename(node.path)] = node

    def to_ypath(self, client=None):
        return yt.YPath(self.path, attributes={"creation_time": self.creation_time}, client=client)


ResultOfYtGet = collections.namedtuple("ResultOfYtGet", ["attributes"])


class MockYtClient(object):
    def __init__(self, nodes):
        self.config = {"prefix": "//"}
        self.root = Node("", "")
        self.index = {}
        for node in nodes:
            path = node.path.split("/")
            subpath = None
            for item in path[:-1]:
                subpath = item if subpath is None else yt.ypath_join(subpath, item)
                if subpath in self.index:
                    self.index[subpath].creation_time = min(self.index[subpath].creation_time, node.creation_time)
                else:
                    self._insert_node(Node(subpath, node.creation_time, "map_node"))

            self._insert_node(node)

    def _insert_node(self, node):
        self.index[node.path] = node
        self._get_parent_node(node.path).add(node)

    def _get_parent_node(self, path):
        if "/" not in path:
            return self.root
        else:
            return self.index[os.path.dirname(path)]

    def _get_attr_dict(self, path, attr_list):
        node_attr_dict = self.index[path].__dict__
        return ResultOfYtGet(
            attributes={
                attr: node_attr_dict[attr] for attr in attr_list if attr in node_attr_dict
            }
        )

    def get(self, path, **kwargs):
        path_parts = path.rsplit("/", 1)
        on_path_attr = None
        if len(path_parts) > 1 and path_parts[-1].startswith("@"):
            path = path_parts[0]
            on_path_attr = path_parts[-1][1:]

        if on_path_attr is not None:
            return self.index[path].get(on_path_attr)
        elif "attributes" in kwargs:
            return self._get_attr_dict(path, kwargs.get("attributes", []))
        else:
            raise Exception("Getting a node itself is not supported")

    def search(self, path, attributes, path_filter, **kwargs):
        path_node = self.index[path]
        assert path_node.type == "map_node"

        def filter_nodes(dict_items):
            return [
                yt.YPath(node.path, attributes={attr: node.get(attr) for attr in attributes}, client=self)
                for name, node in dict_items
                if path_filter(node.path)
            ]

        return filter_nodes([(path, path_node)]) + filter_nodes(path_node.children.items())

    def exists(self, path):
        return path in self.index
