import argparse
import collections
import logging

from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.spine import deploy
from crypta.spine import spine
from crypta.spine.pushers.update_deploy_stages import lib
from crypta.spine.pushers.update_deploy_stages.lib.client import Client


logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser(description="Update deploy stages")
    parser.add_argument("--dry-run", action="store_true", help="Do not apply updates")
    return parser.parse_args()


def main():
    logging_helpers.configure_stderr_logger(logger)

    args = parse_args()

    registry = spine.get_config_registry()

    client = Client()

    stages = registry.get_configs(deploy.Stage.TAG)
    check_duplicates(stages)

    release_rules = registry.get_configs(deploy.DockerReleaseRule.TAG)
    check_duplicates(release_rules)

    for release_rule in release_rules:
        release_rule.fill_urls(client)

    for stage in stages:
        lib.put_stage(stage.proto, client, dry_run=args.dry_run)

    for release_rule in release_rules:
        lib.put_release_rule(release_rule.proto, client, dry_run=args.dry_run)


def check_duplicates(items):
    counts = collections.Counter(x.proto.meta.id for x in items)
    if counts and (counts.most_common(1)[0][1] > 1):
        raise Exception("Duplicate items: {}".format(",".join(str(k) for k in counts if counts[k] > 1)))
