from infra.dctl.src import consts
from infra.dctl.src.lib import (
    cliutil,
    release_rule,
    stage,
    yp_client,
)
from library.python import vault_client
import requests
from yp import (
    common as yp_common,
    data_model,
)

import crypta.lib.python.sandbox.client as sandbox
from crypta.spine.pushers.update_deploy_stages.lib import env


class Client(object):
    def __init__(self, url=consts.XDC.address):
        self.docker_registry_token = env.get_docker_registry_token()
        self.yp_token = env.get_yp_token()

        self.vault_client = vault_client.VaultClient(
            host=vault_client.instances.VAULT_PRODUCTION_API,
            check_status=False,
            authorization=self.yp_token,
        )
        self.vault_client_rsa_fallback = vault_client.VaultClient(
            host=vault_client.instances.VAULT_PRODUCTION_API,
            check_status=False,
            rsa_auth=True,
            rsa_login=cliutil.get_user()
        )
        self.yp_client = yp_client.YpClient(
            url=url,
            token=self.yp_token,
        )
        self.sandbox_client = sandbox.SandboxClient(token=env.get_sandbox_token())

    def get_docker_tag(self, host, image_name):
        response = requests.get(
            "https://{}/v2/{}/tags/list".format(host, image_name),
            headers={"Authorization": "OAuth {}".format(self.docker_registry_token)},
        )

        response.raise_for_status()

        return response.json()["tags"][0]

    def get_stage(self, stage_id):
        try:
            old_stage = self.yp_client.get(
                object_type=data_model.OT_STAGE,
                object_id=stage_id,
            )
            cliutil.clear_not_initializable_fields(old_stage)
            return old_stage
        except yp_common.YpNoSuchObjectError:
            return None

    def push_stage(self, proto):
        stage.put(
            stage=proto,
            cluster=consts.XDC.name,
            rewrite_delegation_tokens=False,
            vault_client=self.vault_client,
            vault_client_rsa_fallback=self.vault_client_rsa_fallback,
            client=self.yp_client,
        )
        print("Stage {} was pushed successfully".format(proto.meta.id))

    def get_release_rule(self, release_rule_id):
        try:
            old_release_rule = self.yp_client.get(
                object_type=data_model.OT_RELEASE_RULE,
                object_id=release_rule_id,
            )

            cliutil.clear_not_initializable_fields(old_release_rule)
            return old_release_rule
        except yp_common.YpNoSuchObjectError:
            return None

    def push_release_rule(self, proto):
        release_rule.put(proto, self.yp_client)
        print("Release rule {} was pushed successfully".format(proto.meta.id))

    def get_sandbox_rbtorrent(self, resource_type, release_type):
        resource_id = self.get_sandbox_resource_id(resource_type, release_type)
        return self.sandbox_client.get_rbtorrent(resource_id)

    def get_sandbox_resource_id(self, resource_type, release_type):
        return self.sandbox_client.get_last_released_resource_id(resource_type, release_type)
