import copy
import datetime
import logging

import dateutil.parser
import jsondiff

from crypta.lib.python.spine.sandbox import (
    consts,
    sandbox_scheduler,
)

logger = logging.getLogger(__name__)


def calc_diff(old, new):
    def canonize(config):
        new_config = copy.deepcopy(config)
        new_config["task"][sandbox_scheduler.CUSTOM_FIELDS] = {x["name"]: x["value"] for x in config["task"][sandbox_scheduler.CUSTOM_FIELDS]}
        return new_config

    def recursive_add_keys(from_, to, is_custom_fields):
        for key in from_:
            if key not in to or to[key] is None:
                to[key] = from_[key]
            elif not is_custom_fields and isinstance(to[key], dict) and isinstance(from_[key], dict):
                recursive_add_keys(from_[key], to[key], key == sandbox_scheduler.CUSTOM_FIELDS)

    old = canonize(old)
    new = canonize(new)

    recursive_add_keys(old, new, False)
    return jsondiff.diff(old, new)


def upload_schedulers(client, tags, schedulers, dry_run):
    existing = client.scheduler.read(limit=3000, tags=tags)

    def item_with_tags(item, tags):
        new_item = copy.deepcopy(item)
        new_item["data"]["task"]["tags"] = tags + item["data"]["task"]["tags"]
        return new_item

    existing_index = {
        (item["task"]["type"], sandbox_scheduler.get_id_from_description(item["task"]["description"])): item
        for item in existing["items"]
        if consts.FIXED_ID not in item["task"]["tags"]
    }
    uploading_index = {
        (item["task_type"], sandbox_scheduler.get_id_from_description(item["data"]["task"]["description"])): item_with_tags(item, tags)
        for item in schedulers
        if item.scheduler_id is None
    }
    existing_index_with_fixed_ids = {
        item["id"]: item
        for item in existing["items"]
        if consts.FIXED_ID in item["task"]["tags"]
    }
    uploading_index_with_fixed_ids = {
        item.scheduler_id: item_with_tags(item, tags)
        for item in schedulers
        if item.scheduler_id is not None
    }

    new_schedulers = set(uploading_index) - set(existing_index)

    for key in new_schedulers:
        id_ = 0
        if not dry_run:
            src_scheduler = uploading_index[key]
            scheduler = client.scheduler.create(src_scheduler)
            id_ = scheduler["id"]
            if src_scheduler.start_immediately:
                client.batch.schedulers.start.update(id_)

        logger.info("Created %s, id: %s", key, id_)

    new_schedulers_with_fixed_id = set(uploading_index_with_fixed_ids) - set(existing_index_with_fixed_ids)
    for key in new_schedulers_with_fixed_id:
        logger.error("Scheduler not found: %s", key)

    update_schedulers(client, existing_index, uploading_index, dry_run)
    update_schedulers(client, existing_index_with_fixed_ids, uploading_index_with_fixed_ids, dry_run)

    now = datetime.datetime.utcnow()

    remove_schedulers(client, existing_index, uploading_index, dry_run, now)
    remove_schedulers(client, existing_index_with_fixed_ids, uploading_index_with_fixed_ids, dry_run, now)
    if new_schedulers_with_fixed_id:
        raise Exception("There were schedulers with fixed ids that were missing from sandbox")


def update_schedulers(client, existing_index, uploading_index, dry_run):
    updated_schedulers = set(existing_index) & set(uploading_index)

    for key in updated_schedulers:
        id_ = existing_index[key]["id"]
        old_scheduler = client.scheduler[id_].read()
        diff = calc_diff(old_scheduler, uploading_index[key]["data"])

        if not dry_run and diff:
            client.scheduler[id_].update(uploading_index[key]["data"])

        if diff:
            logger.info("Updated %s, id: %s", key, id_)
            logger.info("Diff: %s", diff)
        else:
            logger.info("Remained unchanged %s, id: %s", key, id_)


def remove_schedulers(client, existing_index, uploading_index, dry_run, now):
    removed_schedulers = set(existing_index) - set(uploading_index)

    for key in removed_schedulers:
        id_ = existing_index[key]["id"]
        last_run = existing_index[key]["time"]["last"]
        parsed_last_run = dateutil.parser.parse(last_run, ignoretz=True) if last_run is not None else None
        if (parsed_last_run is None) or ((now - parsed_last_run) > datetime.timedelta(days=7)):
            if not dry_run:
                client.batch.schedulers["delete"].update(id_)
            logger.info("Removed %s, id: %s", key, id_)
        else:
            if not dry_run:
                client.batch.schedulers.stop.update(id_)
            logger.info("Stopped %s, id: %s", key, id_)
