import argparse
import collections
import json
import logging
import operator
import os
import sys

from google.protobuf import json_format
import requests

from crypta.lib.python.solomon.client import SolomonClient
from crypta.spine.pushers.upload_solomon_alerts import lib
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.spine.solomon.solomon_alert_registry import SolomonAlertRegistry
from crypta.spine import spine


def parse_args():
    parser = argparse.ArgumentParser(description="Uploads solomon alerts")
    parser.add_argument("--dry-run", action="store_true", help="Print all alerts")
    return parser.parse_args()


def main():
    logger = logging.getLogger()
    logging_helpers.configure_stderr_logger(logger)

    args = parse_args()

    alerts = spine.get_config_registry().get_configs(SolomonAlertRegistry.REGISTRY_TAG)

    counts = collections.Counter((x.projectId, x.name) for x in alerts)
    if counts.most_common(1)[0][1] > 1:
        raise Exception("Duplicate items: {}".format(",".join(str(k) for k in counts if counts[k] > 1)))

    if args.dry_run:
        print json.dumps(
            sorted([json_format.MessageToDict(alert) for alert in alerts], key=operator.itemgetter("projectId", "id")),
            indent=4,
            sort_keys=True,
        )
    else:
        try:
            client = SolomonClient(auth_token=os.environ["SOLOMON_TOKEN"])
            lib.upload_alerts(client, alerts)
        except requests.exceptions.HTTPError as e:
            logger.exception("%s", e.response.text)
            sys.exit(1)
        except Exception:
            logger.exception("Unexpected exception")
            sys.exit(1)
