"""
Get time passed since next expected output table(s) could theoretically appear. Store that time as metric(s) in Solomon.
Examples:
 * Output directory is empty (does not contain tables with names of date format YYYY-MM-DD). No metrics will be sent.
 * Now is some time at 2021-06-01, last created table is //output/dir/2021-05-31 (created at 2021-06-01THH:MM:SS).
   In this case, next expected output table is 2021-05-31 because output table for 2021-06-01 can only appear after 2021-06-01 ends.
   The metric would be equal to (HH * 3600 + MM * 60 + SS).
 * Now is some time at 2021-06-01, last created output table is //output/dir/2021-06-01.
   In this case, the presence of a daily output table of the same date is a logical error,
   and such a table will not be taken into consideration.
 * Now is 2021-06-02:HH:MM:SS, last created output table is //output/dir/2021-05-31.
   In this case, next expected output table is 2021-06-01 because it is the last date after 2021-05-31 and before 2021-06-02 for which we are able
   to produce output table. The metric would be equal to (HH * 3600 + MM * 60 + SS).
 * Now is 2021-06-03:HH:MM:SS, last output tables is the same as previous.
   In this case, next expected output table is again 2021-06-01 because we are still waiting for it to appear.
"""

import argparse
import functools
import json
import logging
import sys

from crypta.lib.python import (
    time_utils,
    yaml_config,
)
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python.spine.yt.yt_output_table_latency_metric import YtOutputTableLatencyMetric
from crypta.lib.python.yt import metrics
from crypta.spine import spine
from crypta.spine.pushers.yt_output_table_latencies import lib


logger = logging.getLogger(__name__)


def parse_args(description=None):
    parser = argparse.ArgumentParser(description=description)
    parser.add_argument("--config", required=True)
    parser.add_argument("--dry-run", action="store_true", help="Print config to stderr")
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args(__doc__)
    logger.info(args)

    config = yaml_config.load(args.config)
    logger.info(config)

    yt_output_table_latency_metrics_config = spine.get_config_registry().get_configs(YtOutputTableLatencyMetric.REGISTRY_TAG)

    if args.dry_run:
        print >> sys.stderr, json.dumps(sorted(yt_output_table_latency_metrics_config), indent=4)
        return

    get_metrics = functools.partial(lib.get_metrics, datetime_now=time_utils.get_current_moscow_datetime())
    metrics.process_yt_metrics(yt_output_table_latency_metrics_config, get_metrics, config["solomon"])
