import logging

import yt.wrapper as yt


logger = logging.getLogger(__name__)


def get_attribute(yt_client, node, attribute, default=None):
    yt_path = yt.ypath_join(node, "@" + attribute)
    if not yt_client.exists(yt_path):
        return default
    return yt_client.get(yt_path)


def _get_node_metrics(node):
    resource_usage = node.attributes.get("resource_usage", {})
    metrics = {
        "disk_space": resource_usage.get("disk_space", 0),
        "chunk_count": resource_usage.get("chunk_count", 0),
        "node_count": resource_usage.get("node_count", 0),
    }

    if node.attributes.get("dynamic", False):
        metrics["unmerged_row_count"] = node.attributes.get("unmerged_row_count", 0)
    else:
        metrics["row_count"] = node.attributes.get("row_count", 0)
    return metrics


def get_metrics(yt_client, node):
    node_type = yt_client.get_type(node)
    if node_type == "table":
        logger.info("Found table")

        class NodeWrapper:
            attributes = {
                "resource_usage": get_attribute(yt_client, node, "resource_usage", {}),
                "dynamic": get_attribute(yt_client, node, "dynamic", False),
                "unmerged_row_count": get_attribute(yt_client, node, "unmerged_row_count", 0),
                "row_count": get_attribute(yt_client, node, "row_count", 0),
            }

        metrics = _get_node_metrics(NodeWrapper())

    elif node_type == "map_node":
        logger.info("Found map_node")
        resource_usage = get_attribute(yt_client, node, "recursive_resource_usage", {})
        metrics = {
            "disk_space": resource_usage.get("disk_space", 0),
            "chunk_count": resource_usage.get("chunk_count", 0),
            "node_count": resource_usage.get("node_count", 0),
        }
    else:
        metrics = {}

    return metrics


def get_recursive_metrics(yt_client, node):
    if yt_client.get_type(node) != "map_node":
        return

    attributes = ("row_count", "unmerged_row_count", "dynamic", "resource_usage")
    for table in yt_client.search(node, node_type=("table",), attributes=attributes):
        yield str(table), _get_node_metrics(table)
