#include <crypta/lib/native/dns/caching_reverse_dns_resolver.h>
#include <crypta/lib/native/http/log_entry_builder_factory.h>
#include <crypta/lib/native/http/request_reply.h>
#include <crypta/lib/native/http/server.h>
#include <crypta/lib/native/juggler/juggler_client.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/logger.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/producer.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/graphite/graphite_stats_sink.h>
#include <crypta/lib/native/stats/log/log_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/stats/utils.h>
#include <crypta/lib/native/tvm/create_tvm_client.h>
#include <crypta/lib/native/tvm/tvm_status_reporter.h>
#include <crypta/lib/native/yaml/parse_yaml_file.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/native/yt/dyntables/async_database/kv_async_database.h>
#include <crypta/lib/native/yt/dyntables/async_database/logger.h>
#include <crypta/styx/services/api/lib/config/clients/clients_config.pb.h>
#include <crypta/styx/services/api/lib/config/config.pb.h>
#include <crypta/styx/services/api/lib/logic/common/mutation_sender/mutation_sender.h>
#include <crypta/styx/services/api/lib/worker/worker.h>

#include <library/cpp/getopt/last_getopt.h>
#include <yt/yt/core/misc/shutdown.h>

#include <util/network/init.h>

using namespace NCrypta;
using namespace NCrypta::NStyx;
using namespace NCrypta::NStyx::NApi;
using namespace NCrypta::NYtDynTables;

namespace {
    struct TAllConfigs {
        TConfig AppConfig;
        TClientsConfig ClientsConfig;
    };

    TAllConfigs GetConfigsFromCmdArgs(int argc, char** argv) {
        TString appConfigFile;
        TString clientsConfigFile;

        auto opts = NLastGetopt::TOpts::Default();
        opts.AddHelpOption();
        opts.AddLongOption("config").StoreResult(&appConfigFile).RequiredArgument("CONFIG").Required();
        opts.AddLongOption("clients").StoreResult(&clientsConfigFile).RequiredArgument("CLIENTS").Required();
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);

        TAllConfigs result;
        result.AppConfig = Yaml2Proto<TConfig>(ParseYamlFile(appConfigFile));
        result.ClientsConfig = Yaml2Proto<TClientsConfig>(ParseYamlFile(clientsConfigFile));

        return result;
    }

    TKvDatabase GetKvDatabase(const TRetryingSingleTableConfig& ytConfig, TStats& kvStats) {
        auto mainDatabase = MakeAtomicShared<TKvAsyncDatabase>(ytConfig.GetClient(), ytConfig.GetMainTable(), ytConfig.GetModifyRows());

        TVector<TKvAsyncDatabasePtr> retryDatabases;
        for (const auto& retryTable : ytConfig.GetRetryTables()) {
            retryDatabases.push_back(MakeAtomicShared<TKvAsyncDatabase>(ytConfig.GetClient(), retryTable, ytConfig.GetModifyRows()));
        }

        return TKvDatabase(mainDatabase, kvStats, retryDatabases, TDuration::MilliSeconds(ytConfig.GetRetryTimeoutMs()));
    }
}

int main(int argc, char** argv) {
    TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

    auto stderrLog = NLog::GetLog();
    try {
        const auto& allConfigs = GetConfigsFromCmdArgs(argc, argv);
        const auto& appConfig = allConfigs.AppConfig;

        NCrypta::NLog::RegisterLogs(appConfig.GetLogs());
        auto log = NLog::GetLog("main");

        InitNetworkSubSystem();

        const auto& ytConfig = appConfig.GetYt();
        NYtDynTables::NLogger::Configure(ytConfig.GetLogger());

        const auto& statsSettings = GetStatsSettings(appConfig.GetStats());

        const auto tvmClient = std::make_shared<NTvmAuth::TTvmClient>(CreateTvmClient(appConfig.GetTvm()));

        TJugglerClient jugglerClient(appConfig.GetJuggler());
        TTvmStatusReporter tvmStatusReporter(*tvmClient, jugglerClient);
        tvmStatusReporter.Start();

        TVector<THolder<IStatsSink>> sinks;
        if (appConfig.HasGraphite()) {
            sinks.emplace_back(MakeHolder<TGraphiteStatsSink>(appConfig.GetGraphite()));
        }
        if (appConfig.HasSolomon()) {
            sinks.emplace_back(MakeHolder<TSolomonStatsSink>(appConfig.GetSolomon()));
        }
        sinks.emplace_back(MakeHolder<TLogStatsSink>("graphite"));

        auto statsWriter = GetStatsWriter(std::move(sinks));

        TStats kvStats("kv", statsSettings);
        auto database = GetKvDatabase(ytConfig, kvStats);

        const auto& logbrokerConfig = appConfig.GetLogbroker();

        const auto& accessLogProducerConfig = logbrokerConfig.GetAccessLogProducer();
        NPQ::TProducer accessLogProducer(
                NPQ::NPQLib::Create(logbrokerConfig.GetPqlib()),
                accessLogProducerConfig,
                logbrokerConfig.GetUseTvm()
                ? NPersQueue::CreateTVMCredentialsProvider(tvmClient, NPQ::NLogger::Create(accessLogProducerConfig.GetLogName()))
                : NPersQueue::CreateInsecureCredentialsProvider(),
                statsSettings
        );

        const auto& changeLogProducerConfig = logbrokerConfig.GetChangeLogProducer();
        NPQ::TProducer changeLogProducer(
                NPQ::NPQLib::Create(logbrokerConfig.GetPqlib()),
                changeLogProducerConfig,
                logbrokerConfig.GetUseTvm()
                    ? NPersQueue::CreateTVMCredentialsProvider(tvmClient, NPQ::NLogger::Create(changeLogProducerConfig.GetLogName()))
                    : NPersQueue::CreateInsecureCredentialsProvider(),
                statsSettings
        );

        TMutationSender mutationSender(changeLogProducer);

        TWorker worker(
                database,
                mutationSender,
                TDuration::Seconds(appConfig.GetMinDeleteIntervalSec()),
                statsSettings,
                appConfig.GetPing(),
                allConfigs.ClientsConfig,
                *tvmClient
        );

        TCachingReverseDnsResolver reverseDnsResolver;
        auto serverProcessFunc = [&worker](NHttp::TRequestReply& reply) {
            worker.ProcessRequest(reply);
        };

        log->info("Starting server on port {} ...", appConfig.GetHttp().GetPort());
        NCrypta::NHttp::TServer server(
                serverProcessFunc,
                appConfig.GetHttp(),
                *tvmClient,
                reverseDnsResolver,
                NewJsonLogEntryBuilderFactory(),
                accessLogProducer,
                statsSettings);

        log->info("Started");

        stopSignalWaiter.Wait();
    } catch (const std::exception& e) {
        stderrLog->error("Exception in main(): {}", e.what());
    } catch (...) {
        stderrLog->error("Unknown exception in main()");
    }

    stderrLog->info("Shutting down YT library...");
    NYT::Shutdown();

    stderrLog->info("Exiting");
    return 0;
}
