#include "delete_processor_base.h"

#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/styx/services/api/lib/logic/common/response/delete_response.pb.h>
#include <crypta/styx/services/api/lib/logic/common/response_serializers/delete_response_serializer.h>

#include <library/cpp/svnversion/svnversion.h>

#include <util/string/builder.h>
#include <util/string/cast.h>

using namespace NCrypta::NStyx::NApi;

TDeleteProcessorBase::TDeleteProcessorBase(TMutationSender& mutationSender, const TString& handlePath, NLog::TLogPtr log, TStats& stats)
        : TRequestProcessor(log, stats)
        , MutationSender(mutationSender)
        , HandlePath(handlePath)
{
}

void TDeleteProcessorBase::DoProcess(NHttp::TRequestReply& reply, const TClient& clientInfo) {
    TScopeTimer scopeTimer(Stats.Percentile, "timing.process");

    Stats.Count->Add("request.total.received");
    Stats.Count->Add("tvm_client." + clientInfo.GetName());

    TDeleteRequest request;
    try {
        ParseRequest(reply, request);
    } catch (const yexception& e) {
        Log->error("Error while parsing request: {}. Request body = {}", e.what(), reply.GetRequestBody());
        SendResponse(reply, HTTP_BAD_REQUEST, e.what());
        return;
    }
    Stats.Count->Add("subclient." + clientInfo.GetName() + "." + request.Subclient);

    if (request.ServiceIds.empty()) {
        RespondError(reply, "no_service_ids", "No service ids in request");
        return;
    }

    for (const auto& serviceId : request.ServiceIds) {
        if (serviceId != "matching") {
            RespondError(reply, "unknown_service_id", TString("Unknown service_id: ") + serviceId);
            return;
        }
    }

    MutationSender.DeletePuid(request.DefaultPuid);

    Respond(reply);
}

void TDeleteProcessorBase::Respond(NHttp::TRequestReply& reply) {
    TDeleteResponse response;
    response.SetStatus("ok");
    Stats.Count->Add("takeout_status.ok");

    SendResponse(reply, HTTP_OK, NDeleteResponseSerializer::ToString(response));
}

void TDeleteProcessorBase::RespondError(NHttp::TRequestReply& reply, const TString& code, const TString& message) {
    TDeleteResponse response;
    response.SetStatus("error");
    Stats.Count->Add("takeout_status.error");

    auto* error = response.AddErrors();
    error->SetCode(code);
    error->SetMessage(message);

    SendResponse(reply, HTTP_OK, NDeleteResponseSerializer::ToString(response));
}
