#include "status_processor_base.h"

#include <crypta/lib/native/graphite/utils.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/styx/services/api/lib/logic/common/response/status_response.pb.h>
#include <crypta/styx/services/api/lib/logic/common/response_serializers/status_response_serializer.h>
#include <crypta/styx/services/common/data/puid_id.h>
#include <crypta/styx/services/common/db_state/db_state_loader.h>

#include <library/cpp/svnversion/svnversion.h>

#include <util/string/builder.h>
#include <util/string/cast.h>

using namespace NCrypta::NStyx;
using namespace NCrypta::NStyx::NApi;

namespace {
    const TString READY_TO_DELETE_STATUS = "ready_to_delete";
    const TString DELETE_IN_PROGRESS_STATUS = "delete_in_progress";

    TString GetPuidState(ui64 puid, const TDbState::TPuidStates& puidStates, TDuration minDeleteInterval, TInstant now) {
        auto puidStateIt = puidStates.find(puid);
        if (puidStateIt == puidStates.end()) {
            return READY_TO_DELETE_STATUS;
        }

        const auto& puidState = puidStateIt->second;
        const auto oblivionEventsSize = puidState.OblivionEventsSize();
        if (oblivionEventsSize == 0) {
            return READY_TO_DELETE_STATUS;
        }

        const auto& lastOblivionEvent = puidState.GetOblivionEvents(oblivionEventsSize - 1);
        if (now - TInstant::Seconds(lastOblivionEvent.GetUnixtime()) >= minDeleteInterval) {
            return READY_TO_DELETE_STATUS;
        } else {
            return DELETE_IN_PROGRESS_STATUS;
        }
    }
}

TStatusProcessorBase::TStatusProcessorBase(
        NYtDynTables::TKvDatabase& replicaDatabase,
        TDuration minDeleteInterval,
        const TString& handlePath,
        NLog::TLogPtr log,
        TStats& stats)
    : TRequestProcessor(log, stats)
    , ReplicaDatabase(replicaDatabase)
    , MinDeleteInterval(minDeleteInterval)
    , HandlePath(handlePath)
{
}

void TStatusProcessorBase::DoProcess(NHttp::TRequestReply& reply, const TClient& clientInfo) {
    TScopeTimer scopeTimer(Stats.Percentile, "timing.process");

    Stats.Count->Add("request.total.received");
    Stats.Count->Add(MakeGraphiteMetric("tvm_client", clientInfo.GetName()));

    TRequestWithPuids request;
    try {
        ParseRequest(reply, request);
    } catch (const yexception& e) {
        Log->error("Error while parsing request: {}", e.what());
        SendResponse(reply, HTTP_BAD_REQUEST, e.what());
        return;
    }
    Stats.Count->Add(MakeGraphiteMetric("subclient", clientInfo.GetName(), request.Subclient));

    if (request.DefaultPuid != 0) {
        const auto dbState = TDbStateLoader(Log, TDuration()).Load(ReplicaDatabase, {NStyx::GetPuid(request.DefaultPuid)});
        
        Respond(reply, request.DefaultPuid, dbState.GetPuidStates());
    } else {
        RespondError(reply, "Default puid is zero");
    }
}

void TStatusProcessorBase::Respond(NHttp::TRequestReply& reply, ui64 defaultPuid, const TDbState::TPuidStates& puidStates) {
    TStatusResponse response;
    response.SetStatus("ok");
    Stats.Count->Add("takeout_status.ok");

    auto* responseItem = response.AddData();
    responseItem->SetId("matching");
    responseItem->SetSlug("matching");
    responseItem->SetState(GetPuidState(defaultPuid, puidStates, MinDeleteInterval, TShiftedClock::Now()));

    SendResponse(reply, HTTP_OK, NStatusResponseSerializer::ToString(response));
}

void TStatusProcessorBase::RespondError(NHttp::TRequestReply& reply, const TString& message) {
    TStatusResponse response;
    response.SetStatus("error");
    Stats.Count->Add("takeout_status.error");

    auto* error = response.AddErrors();
    error->SetCode("zero_default_puid");
    error->SetMessage(message);

    SendResponse(reply, HTTP_OK, NStatusResponseSerializer::ToString(response));
}
