#include "request_processor_factory.h"

#include <crypta/styx/services/api/lib/logic/delete_private/delete_private_processor.h>
#include <crypta/styx/services/api/lib/logic/delete/delete_processor.h>
#include <crypta/styx/services/api/lib/logic/ping/ping_processor.h>
#include <crypta/styx/services/api/lib/logic/status_private/status_private_processor.h>
#include <crypta/styx/services/api/lib/logic/status/status_processor.h>
#include <crypta/styx/services/api/lib/logic/version/version_processor.h>

#include <library/cpp/http/misc/parsed_request.h>

#include <util/generic/hash.h>
#include <util/generic/ptr.h>


using namespace NCrypta::NStyx;
using namespace NCrypta::NStyx::NApi;

namespace {
    struct THandle {
        TString Method;
        TString Path;

        THandle(TStringBuf method, TStringBuf path)
            : Method(method)
            , Path(path)
        {
        }

        bool operator==(const THandle&) const = default;
    };
}

template <>
struct THash<THandle> {
    size_t operator()(const THandle& handle) {
        return ComputeHash(handle.Method) ^ ComputeHash(handle.Path);
    }
};

namespace {
    const THandle STATUS_HANDLE("GET", "1/takeout/status/");
    const THandle DELETE_HANDLE("POST", "1/takeout/delete/");
    const THandle PING_HANDLE("GET", "ping");
    const THandle VERSION_HANDLE("GET", "version");

    // For initial debug purposes, to be deleted before release:
    const THandle STATUS_PRIVATE_HANDLE("GET", "status_private");
    const THandle DELETE_PRIVATE_HANDLE("GET", "delete_private");

    using TProcessorMap = THashMap<THandle, THolder<IRequestProcessor>>;

    TProcessorMap GetProcessorMap(
            NCrypta::NYtDynTables::TKvDatabase& replicaDatabase,
            TDuration minDeleteInterval,
            TMutationSender& mutationSender,
            const TStats::TSettings& statsSettings,
            const TPingConfig& pingConfig,
            const NTvmAuth::TTvmClient& tvmClient)
    {
        TProcessorMap result;
        result[STATUS_HANDLE] = MakeHolder<TStatusProcessor>(replicaDatabase, minDeleteInterval, statsSettings);
        result[STATUS_PRIVATE_HANDLE] = MakeHolder<TStatusPrivateProcessor>(replicaDatabase, minDeleteInterval, statsSettings);
        result[DELETE_HANDLE] =  MakeHolder<TDeleteProcessor>(mutationSender, statsSettings);
        result[DELETE_PRIVATE_HANDLE] =  MakeHolder<TDeletePrivateProcessor>(mutationSender, statsSettings);
        result[PING_HANDLE] =  MakeHolder<TPingProcessor>(pingConfig, tvmClient, statsSettings);
        result[VERSION_HANDLE] =  MakeHolder<TVersionProcessor>(statsSettings);

        return result;
    }

    bool IsClientAllowed(TStringBuf method, const TClient& client) {
        const auto& permissions = client.GetPermissions();
        if (method == STATUS_HANDLE.Path) {
            return permissions.GetStatusAllowed();
        } else if (method == DELETE_HANDLE.Path) {
            return permissions.GetDeleteAllowed();
        } else if (method == PING_HANDLE.Path) {
            return permissions.GetPingAllowed();
        } else if (method == VERSION_HANDLE.Path) {
            return permissions.GetVersionAllowed();
        } else if (method == STATUS_PRIVATE_HANDLE.Path) {
            return true;
        } else if (method == DELETE_PRIVATE_HANDLE.Path) {
            return true;
        }
        return false;
    }
}

TProcessorFactory::TProcessorFactory(
        NYtDynTables::TKvDatabase& replicaDatabase,
        TMutationSender& mutationSender,
        TDuration minDeleteInterval,
        const TStats::TSettings& statsSettings,
        const TPingConfig& pingConfig,
        const NTvmAuth::TTvmClient& tvmClient)
        : ReplicaDatabase(replicaDatabase)
        , MutationSender(mutationSender)
        , MinDeleteInterval(minDeleteInterval)
        , StatsSettings(statsSettings)
        , PingConfig(pingConfig)
        , TvmClient(tvmClient)
{
    Y_UNUSED(ReplicaDatabase);
    Y_UNUSED(MutationSender);
}

IRequestProcessor* TProcessorFactory::GetProcessor(const TStringBuf& method, const TStringBuf& service, const TClient& client) {
    static TProcessorMap processorMap = GetProcessorMap(ReplicaDatabase, MinDeleteInterval, MutationSender, StatsSettings, PingConfig, TvmClient);

    if (auto processor = processorMap.find(THandle(method, service)); processor != processorMap.end()) {
        if (!IsClientAllowed(service, client)) {
            ythrow TNotAllowedException();
        }
        return processor->second.Get();
    }

    return nullptr;
}
