import collections
from datetime import timedelta as dt

from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler.aggregate_hosts import AggregateHosts
from crypta.lib.python.spine.juggler import juggler_check_generator
from crypta.lib.python.spine.sandbox import sandbox_scheduler

COMMON_AGGR_HOSTS = AggregateHosts(testing_host="crypta_styx_testing", production_host="crypta_styx")
PROJECT = "crypta_styx"

seneca_vla_plus_hahn = [YtProxy.seneca_vla] + list(YtProxy.Group.offline)
OFFLINE_AND_VLA_DYNTABLE_YT_DEPENDENCIES = {
    environment.STABLE: seneca_vla_plus_hahn,
    environment.TESTING: seneca_vla_plus_hahn,
}

UniversalTask = collections.namedtuple("UniversalTask", ["bundle_name", "cmd", "schedule_interval", "kill_timeout", "crit_time", "aggr_hosts", "yt_deps"])


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-styx"])

    add_sandbox_tasks(juggler)

    return juggler


def add_sandbox_tasks(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["STYX"])

    for task in [
        UniversalTask(
            bundle_name="styx-history-publisher",
            cmd=["{{cwd}}/crypta-styx-history-publisher", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(hours=24),
            kill_timeout=dt(hours=2),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host, COMMON_AGGR_HOSTS.testing_host],
            yt_deps=OFFLINE_AND_VLA_DYNTABLE_YT_DEPENDENCIES,
        ),
        UniversalTask(
            bundle_name="styx-raw-events-publisher",
            cmd=["{{cwd}}/crypta-styx-raw-events-publisher", "--config", "{{cwd}}/config.yaml"],
            schedule_interval=dt(hours=24),
            kill_timeout=dt(hours=2),
            crit_time=dt(days=2),
            aggr_hosts=[COMMON_AGGR_HOSTS.production_host, COMMON_AGGR_HOSTS.testing_host],
            yt_deps=OFFLINE_AND_VLA_DYNTABLE_YT_DEPENDENCIES,
        ),
    ]:
        for aggr_host in task.aggr_hosts:
            sandbox.create_run_universal_bundle_scheduler(
                bundle_name=task.bundle_name,
                cmd=task.cmd,
                env=aggr_host.environment,
                schedule_interval=task.schedule_interval,
                kill_timeout=task.kill_timeout,
            ).check(crit_time=task.crit_time).set_host(aggr_host.host).add_yt_dependencies(task.yt_deps[aggr_host.environment])
