import datetime
import os

import library.python.resource as rs
import yt.wrapper as yt
from yt.wrapper import ypath

from crypta.lib.python import time_utils
from crypta.lib.python.yql import yql_helpers
import crypta.lib.python.yql.client as yql
from crypta.lib.python.yt import yt_helpers
from crypta.tx.services.common import (
    helpers,
    schema,
)


class Params:
    output_table = "output_table"
    crypta_identifier_udf_url = "crypta_identifier_udf_url"
    yql_libs = [rs.find(os.path.join('/yql_lib', lib)).decode('utf-8') for lib in ['get_timestamp.yql', 'statuses.yql', 'stop_words.yql', 'lemmas.yql']]

    values = (output_table, crypta_identifier_udf_url)


def import_data(config, yql_query, yql_query_params, yql_cache_dir, yql_cache_ttl, output_table, tx, logger):
    yql_query_params[Params.output_table] = output_table
    yql_helpers.run_query(
        yql_query,
        config.Yt,
        yql_query_params,
        logger,
        tx=tx,
        udfs=[yql.Udf("libcrypta_identifier_udf.so", config.CryptaIdentifierUdfUrl)],
        yql_libs=Params.yql_libs,
        binary_cache_tmp_folder=yql_cache_dir,
        binary_cache_ttl=yql_cache_ttl,
    )


def match_to_crypta_id(config, yt_client, yql_cache_dir, yql_cache_ttl, input_table, output_table, tx, logger):
    with yt.TempTable(config.Yt.TmpDir, client=yt_client) as tmp_table:
        yql_helpers.run_query(
            query_template="/query/match_with_crypta_id.yql",
            yt_config=config.Yt,
            query_params={
                "input_table": input_table,
                "output_table": tmp_table,
                "matching_tables": config.MatchingTables,
            },
            logger=logger,
            tx=tx,
            binary_cache_tmp_folder=yql_cache_dir,
            binary_cache_ttl=yql_cache_ttl,
        )

        yt_client.create("table", output_table, recursive=True, force=True, attributes={"schema": schema.get_transaction_schema()})
        yt_client.run_merge(tmp_table, output_table, mode="sorted", spec={"force_transform": True, "combine_chunks": True, "merge_by": schema.TRANSACTION_KEY_COLUMNS})


def calc_stats(config, yql_cache_dir, yql_cache_ttl, data_table, transaction_source, tx, logger):
    period_formats = {
        "daily": "%Y-%m-%d",
        "monthly": "%Y-%m",
    }

    for period, date_format in period_formats.items():
        yql_helpers.run_query(
            query_template="/query/calc_stats.yql",
            yt_config=config.Yt,
            query_params={
                "input_table": data_table,
                "period_format": date_format,
                "output_table": helpers.get_tx_table_path(ypath.ypath_join(config.StatsDir, period), transaction_source),
            },
            logger=logger,
            tx=tx,
            yql_libs=Params.yql_libs,
            binary_cache_tmp_folder=yql_cache_dir,
            binary_cache_ttl=yql_cache_ttl,
        )


def run(config, transaction_source, yql_query, yql_query_params, logger):
    if any(x in yql_query_params for x in Params.values):
        raise ValueError("Don't provide {} in yql_query_params".format(Params.values))

    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)
    yt_client.create("map_node", config.Yt.TmpDir, recursive=True, ignore_existing=True)

    yql_cache_dir = config.Yt.TmpDir
    yql_cache_ttl = datetime.timedelta(seconds=config.YqlBinaryCacheTtlSec)
    # TODO(CRYPTA-16876) YQL creates this dir without tx, need to investigate
    yt_client.create("map_node", yql_cache_dir + "/tmp", recursive=True, ignore_existing=True)

    def get_tmp_data_table():
        attributes = {"schema": schema.get_unordered_transaction_schema()}
        return yt.TempTable(config.Yt.TmpDir, client=yt_client, attributes=attributes)

    with yt_client.Transaction() as tx, get_tmp_data_table() as tmp_data_table:
        import_data(config, yql_query, yql_query_params, yql_cache_dir, yql_cache_ttl, tmp_data_table, tx, logger)

        data_table = helpers.get_tx_table_path(config.DataDir, transaction_source)
        match_to_crypta_id(config, yt_client, yql_cache_dir, yql_cache_ttl, tmp_data_table, data_table, tx, logger)
        yt_client.set_attribute(data_table, helpers.GENERATE_DATE, time_utils.get_current_moscow_datetime().date().isoformat())

        calc_stats(config, yql_cache_dir, yql_cache_ttl, data_table, transaction_source, tx, logger)
