import datetime

import library.python.resource as rs
import pytz
from reactor_client import reactor_objects
from reactor_client.reactor_api import ReactorAPIClientV1
import yt.wrapper as yt

from crypta.lib.python import time_utils
from crypta.lib.python.yql import yql_helpers
from crypta.lib.python.yt import yt_helpers


def get_unprocessed_dates(yt_client, config, yesterday):
    existing_dates = set(yt_client.list(config.OutputDir) if yt_client.exists(config.OutputDir) else [])
    day_step = datetime.timedelta(days=1)
    result = []
    it = yesterday

    while len(result) < config.BatchSize:
        str_date = str(it)
        if str_date < config.MinDate:
            break
        if str_date not in existing_dates:
            result.append(str_date)
        it -= day_step

    return result


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)

    reactor_client = ReactorAPIClientV1(base_url=config.Reactor.Url, token=config.Reactor.Token)
    artifact_id = reactor_objects.ArtifactIdentifier(
        namespace_identifier=reactor_objects.NamespaceIdentifier(namespace_path=config.Artifact)
    )

    yesterday = (time_utils.get_current_moscow_datetime() - datetime.timedelta(days=1)).date()
    yql_libs = [rs.find(path).decode('utf-8') for path in ['/yql_lib/date_lib.yql', '/yql_lib/new_state.yql']]

    yql_cache_dir = config.Yt.TmpDir
    yql_cache_ttl = datetime.timedelta(seconds=config.YqlBinaryCacheTtlSec)
    yt_client.mkdir(yql_cache_dir, recursive=True)

    with yt_client.Transaction() as tx:
        yql_query_params = {
            "input_dir": config.DataDir,
            "state_table": config.StateTable,
            "yesterday": yesterday.isoformat(),
        }
        query = "/query/update_state.yql" if yt_client.exists(config.StateTable) else "/query/create_state.yql"
        yql_helpers.run_query(query, config.Yt, yql_query_params, logger, tx=tx, yql_libs=yql_libs, binary_cache_tmp_folder=yql_cache_dir, binary_cache_ttl=yql_cache_ttl)

    while True:
        with yt_client.Transaction() as tx:
            unprocessed_dates = get_unprocessed_dates(yt_client, config, yesterday)
            logger.info("Unprocessed dates %s", unprocessed_dates)

            if not unprocessed_dates:
                logger.info("No dates to process")
                break

            yql_query_params = {
                "input_dir": config.DataDir,
                "output_dir": config.OutputDir,
                "dates": unprocessed_dates,
                "state_table": config.StateTable,
            }
            yql_helpers.run_query("/query/ltp_logos_export.yql", config.Yt, yql_query_params, logger, tx=tx, yql_libs=yql_libs, binary_cache_tmp_folder=yql_cache_dir, binary_cache_ttl=yql_cache_ttl)

        for date in unprocessed_dates:
            path = yt.ypath_join(config.OutputDir, date)

            ypath_meta = reactor_objects.Metadata(
                type_="/yandex.reactor.artifact.YtPathArtifactValueProto",
                dict_obj={
                    "cluster": "hahn",
                    "path": path,
                },
            )

            moscow_dt = pytz.timezone("Europe/Moscow").localize(datetime.datetime.fromisoformat(date)) + datetime.timedelta(days=1)

            reactor_client.artifact_instance.instantiate(
                artifact_identifier=artifact_id,
                metadata=ypath_meta,
                user_time=moscow_dt,
                create_if_not_exist=True,
            )
