import argparse
import logging
import pprint
import sys

import yt.wrapper as yt

from crypta.utils.providers_index_builder.lib.providers_index_builder import upload_table, download_table


logger = logging.getLogger(__name__)
pp = pprint.PrettyPrinter(indent=4)


def configure_logger():
    logger.setLevel(logging.INFO)
    handler = logging.StreamHandler(stream=sys.stderr)
    formatter = logging.Formatter('[%(asctime)s] [%(levelname)s] %(message)s')
    handler.setFormatter(formatter)
    logger.addHandler(handler)


def init_args():
    parser = argparse.ArgumentParser()

    group = parser.add_mutually_exclusive_group(required=True)
    group.add_argument("--upload", action="store_true", help="Upload config file as public index")
    group.add_argument("--download", action="store_true", help="Download public index into config file")

    parser.add_argument("--config-file", required=True, help="Config file. For --download, use '-' for stdout")
    parser.add_argument("--index-table", required=True, help="Path to public index table")
    parser.add_argument("--yt-proxy", required=True)
    parser.add_argument("--dry-run", action="store_true", help="Make no actual changes")

    return parser.parse_args()


def main():
    configure_logger()
    args = init_args()

    logger.info("args = {}".format(pp.pformat(args)))

    upload = args.upload
    config_file = args.config_file
    index_table = args.index_table
    dry_run = args.dry_run

    if not index_table:
        logger.error("Index table must not be empty")
        exit(1)

    yt.config.set_proxy(args.yt_proxy)

    try:
        if upload:
            ret = upload_table(config_file, index_table, dry_run)
        else:
            ret = download_table(config_file, index_table, dry_run)
        exit(ret)
    except Exception:
        logger.exception("Caught exception when {} table".format("uploading" if upload else "downloading"))
        exit(1)


if __name__ == "__main__":
    main()
