#!/usr/bin/env python
import contextlib
import itertools
import json
import logging
import os
import sys
import yt.wrapper as yt

logger = logging.getLogger(__name__)


def get_all_keys(list_of_dicts):
    return set([k for sublist in list_of_dicts for k in sublist])


class Index:

    @staticmethod
    def validate(rows):
        if type(rows) is not list:
            raise Exception("rows must be list, but got {}".format(rows))

        if not all((type(row) is dict) for row in rows):
            raise Exception("Each entry must be a dict (got {})".format(rows))

        all_keys = get_all_keys(rows)
        for x in rows:
            if set(x) != all_keys:
                raise Exception("Each entry must have identical set of fields (all fields = {}, mismatching entry = {}".format(all_keys, x))
            if ("id" not in x) or (type(x["id"]) not in {int, long}):
                raise Exception("Each entry must have an 'id' key and its value must be of integer type (got {})".format(x))

    def __init__(self, rows):
        self.validate(rows)
        self.rows = sorted(rows, key=lambda x: x["id"])
        self.all_keys = get_all_keys(rows)

    def __eq__(self, other):
        return self.rows == other.rows

    def __ne__(self, other):
        return not self.__eq__(other)

    def get_diff(self, other):
        diff = ""
        if self.all_keys != other.all_keys:
            left = self.all_keys
            right = other.all_keys
            diff = "Different sets of keys: {} != {}\n".format(left, right)
            diff += "Symmetric difference: {}".format(left.symmetric_difference(right))
        elif self.rows != other.rows:
            left = self.rows
            right = other.rows
            sym_diff = list(itertools.ifilterfalse(lambda x: x in left, right))
            sym_diff.extend(list(itertools.ifilterfalse(lambda x: x in right, left)))

            if sym_diff:
                diff = "Different lists of items. Symmetric difference:\n{}".format(sym_diff)

        return diff

    def __str__(self):
        return json.dumps(self.rows, indent=4, sort_keys=True, separators=[",", ": "], ensure_ascii=False)


def read_index_table(table):
    return Index(list(yt.read_table(table)))


def write_index_table(index_table, index):
    yt.mkdir(os.path.dirname(index_table), recursive=True)
    yt.write_table(index_table, index.rows, force_create=True)


@contextlib.contextmanager
def smart_open(filename, mode="r"):
    if filename == '-':
        if mode is None or mode == '' or 'r' in mode:
            fh = sys.stdin
        else:
            fh = sys.stdout
    else:
        fh = open(filename, mode)

    try:
        yield fh
    finally:
        if filename != '-':
            fh.close()


def read_config(config_file):
    with smart_open(config_file) as f:
        rows = json.load(f)
        return Index(rows)


def write_config(config_file, config):
    with smart_open(config_file, "w") as f:
        print >>f, config


def upload_table(config_file, index_table, dry_run):
    config = read_config(config_file)
    public_index = read_index_table(index_table) if yt.exists(index_table) else None

    have_public_index = public_index is not None
    have_difference = not have_public_index or config != public_index

    if have_public_index and have_difference:
        logger.info("Found differences between config file and public index:")
        logger.info(config.get_diff(public_index))
    elif have_public_index:
        logger.info(u"Config file and public index are IDENTICAL:\n{}".format(config))
    else:
        logger.info(u"Found no public index. Config file is:\n{}".format(config))

    if not dry_run:
        logger.info("Performing upload")
        write_index_table(index_table, config)
        logger.info("Successfully uploaded config file {} as public index table {}".format(config_file, index_table))
    else:
        logger.info("Upload public index. Performed dry run")

    return 0


def download_table(config_file, index_table, dry_run):
    if not yt.exists(index_table):
        logger.error("Index table '{}' does not exist".format(index_table))
        return 1

    public_index = read_index_table(index_table)

    if not dry_run:
        logger.info("Download public index")
        write_config(config_file, public_index)
        logger.info("Successfully written public index table {} as file {}".format(index_table, config_file))
    else:
        logger.info(u"Download public index. Performing dry run. Public index:\n{}".format(public_index))

    return 0
