import logging
import socket

from crypta.lib.python.juggler.juggler_queue import JugglerQueue

logger = logging.getLogger(__name__)


class ResourceReleaser(object):
    def __init__(self, resources, db, reports_client, sandbox, version_thresholds, ok_rate_thresholds, sequence, juggler_dry_run):
        self.resources = resources
        self.db = db
        self.reports_client = reports_client
        self.sandbox = sandbox
        self.version_thresholds = version_thresholds
        self.ok_rate_thresholds = ok_rate_thresholds
        self.sequence = sequence
        self.juggler_queue = JugglerQueue(juggler_dry_run)

    def release(self):
        for resource in self.resources:
            for release_type, next_release_type in reversed(zip(self.sequence[:-1], self.sequence[1:])):
                version = self.db.get_latest_resource_version(release_type, resource.name)
                logger.info("Checking %s in %s, version %s", resource.name, release_type, version)
                if version is None:
                    continue

                actual_release_type = self.sandbox.get_release_type(version)
                if actual_release_type != release_type:
                    continue

                counts = self.reports_client.get_report_counts(release_type, resource.name, version)
                logger.info("Counts %s", counts)

                if counts.version < self.version_thresholds[release_type]:
                    continue

                if counts.ok < self.ok_rate_thresholds[release_type] * counts.version:
                    continue

                self.sandbox.release(version, next_release_type, new_ttl=resource.ttls.get(next_release_type))
                self.juggler_queue.enqueue_ok(socket.getfqdn(), "released_{}_{}".format(release_type, resource.name), "OK")
                logger.info("Released to %s", next_release_type)
