package rsserver

import (
	"a.yandex-team.ru/crypta/lib/go/yt/kv"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ytrpc"
	"errors"
	"fmt"
	"github.com/spf13/pflag"
	"gopkg.in/yaml.v2"
	"os"
	"strings"
	"time"
)

type HTTPConfig struct {
	Port       int64   `yaml:"port"`
	TimeoutSec float64 `yaml:"timeout_sec"`
}

type MasterKvClientConfig struct {
	Proxy string `yaml:"proxy"`
	Path  string `yaml:"path"`
}

type ReplicaKvClientsConfig struct {
	DCs            map[string]string `yaml:"dcs"`
	Path           string            `yaml:"path"`
	RetryTimeoutMs int64             `yaml:"retry_timeout_ms"`
}

type Config struct {
	Resources        Resources              `yaml:"resources"`
	FileClient       FileClientConfig       `yaml:"file_client"`
	HTTPConfig       HTTPConfig             `yaml:"http"`
	MasterKVClient   MasterKvClientConfig   `yaml:"master_kv_client"`
	ReplicaKVClients ReplicaKvClientsConfig `yaml:"replica_kv_clients"`
	ReportTTLSec     int64                  `yaml:"report_ttl_sec"`
	SolomonPort      int64                  `yaml:"solomon_port"`
	ReportTimeoutMs  int64                  `yaml:"report_timeout_ms"`
}

func GetConfig() (Config, error) {
	configPath := pflag.String("config", "", "Config file")
	pflag.Parse()

	configFile, err := os.Open(*configPath)
	if err != nil {
		return Config{}, err
	}
	defer configFile.Close()

	var config Config
	if err = yaml.NewDecoder(configFile).Decode(&config); err != nil {
		return Config{}, err
	}

	return config, nil
}

func GetReadOnlyClient(config Config) (kv.ReadOnlyClient, error) {
	currentDC, ok := os.LookupEnv("QLOUD_DATACENTER")
	if !ok {
		currentDC, ok = os.LookupEnv("DEPLOY_NODE_DC")
	}
	if !ok {
		return nil, errors.New("both environment variables QLOUD_DATACENTER and DEPLOY_NODE_DC are undefined")
	}
	currentDC = strings.ToUpper(currentDC)

	replicaConfig := config.ReplicaKVClients
	path := ypath.Path(replicaConfig.Path)

	var mainClient kv.Client
	var retryClients []kv.Client

	for dc, proxy := range replicaConfig.DCs {
		ytClient, err := ytrpc.NewClient(&yt.Config{
			Proxy:             proxy,
			ReadTokenFromFile: true,
		})

		if err != nil {
			return nil, err
		}

		kvClient := kv.NewClient(ytClient, path)
		if dc == currentDC {
			mainClient = kvClient
		} else {
			retryClients = append(retryClients, kvClient)
		}
	}

	if mainClient == nil {
		return nil, fmt.Errorf("proxy for DC=%s not found", currentDC)
	}

	return kv.NewCrossDcClient(mainClient, retryClients, time.Duration(replicaConfig.RetryTimeoutMs)*time.Millisecond), nil
}

func GetMasterClient(config Config) (kv.Client, error) {
	masterConfig := config.MasterKVClient
	ytClient, err := ytrpc.NewClient(&yt.Config{
		Proxy:             masterConfig.Proxy,
		ReadTokenFromFile: true,
	})
	if err != nil {
		return nil, err
	}

	return kv.NewClient(ytClient, ypath.Path(masterConfig.Path)), nil
}
