package rsserver

import (
	"a.yandex-team.ru/crypta/lib/go/yt/kv"
	"context"
	"encoding/json"
	"fmt"
)

const (
	publicResources = "public_resources"
)

type ReadOnlyDBClient interface {
	GetEnv(ctx context.Context, dc string) (*string, error)
	GetPublicResources(ctx context.Context) ([]ResourceID, error)
	GetLatestResourceVersion(ctx context.Context, env string, resource string) (*uint64, error)
}

type DBClient interface {
	ReadOnlyDBClient
	SetEnv(ctx context.Context, dc string, env string) error
	SetPublicResources(ctx context.Context, resources []ResourceID) error
	SetLatestResourceVersion(ctx context.Context, env string, resource string, version uint64) error
}

type readOnlyDBClient struct {
	client kv.ReadOnlyClient
}

type dbClient struct {
	readOnlyDBClient
	client kv.Client
}

type ResourceID struct {
	Name        string
	ReleaseType string
	Version     uint64
}

func (resource *ResourceID) UnmarshalJSON(buf []byte) error {
	tmp := []interface{}{&resource.Name, &resource.ReleaseType, &resource.Version}
	expectedLen := len(tmp)
	if err := json.Unmarshal(buf, &tmp); err != nil {
		return err
	}
	if actualLen := len(tmp); actualLen != expectedLen {
		return fmt.Errorf("wrong number of fields in ResourceID: %d != %d", actualLen, expectedLen)
	}
	return nil
}

func (resource *ResourceID) MarshalJSON() ([]byte, error) {
	return json.Marshal([]interface{}{&resource.Name, &resource.ReleaseType, &resource.Version})
}

func (client *dbClient) SetEnv(ctx context.Context, dc string, env string) error {
	return writeJSON(ctx, client.client, getEnvKey(dc), env)
}

func (client *dbClient) SetPublicResources(ctx context.Context, resources []ResourceID) error {
	return writeJSON(ctx, client.client, publicResources, resources)
}

func (client *dbClient) SetLatestResourceVersion(ctx context.Context, env string, resource string, version uint64) error {
	return writeJSON(ctx, client.client, getResourceKey(env, resource), version)
}

func (client *readOnlyDBClient) GetEnv(ctx context.Context, dc string) (*string, error) {
	var value *string
	return value, lookupJSON(ctx, client.client, getEnvKey(dc), &value)
}

func (client *readOnlyDBClient) GetPublicResources(ctx context.Context) ([]ResourceID, error) {
	var ids []ResourceID
	return ids, lookupJSON(ctx, client.client, publicResources, &ids)
}

func (client *readOnlyDBClient) GetLatestResourceVersion(ctx context.Context, env string, resource string) (*uint64, error) {
	var version *uint64
	return version, lookupJSON(ctx, client.client, getResourceKey(env, resource), &version)
}

func NewReadOnlyDBClient(client kv.ReadOnlyClient) ReadOnlyDBClient {
	return &readOnlyDBClient{client: client}
}

func NewDBClient(client kv.Client) DBClient {
	return &dbClient{
		readOnlyDBClient: readOnlyDBClient{
			client: client,
		},
		client: client,
	}
}

func getEnvKey(dc string) string {
	return getKey("env", dc)
}

func getResourceKey(env string, resource string) string {
	return getKey("resource", env, resource)
}
