package rsserver

import (
	"io/ioutil"
	"os"
	"path/filepath"
	"strconv"
)

type FileClientConfig struct {
	Root string `yaml:"root"`
}

type FileClient interface {
	Init(resources []string) error
	GetResourceDir(name string) string
	GetResourceFilename(resourceName string, resourceVersion uint64) string
	IsPresent(resourceName string, resourceVersion uint64) bool
}

type fileClient struct {
	Root string
}

func (client *fileClient) Init(resources []string) error {
	err := os.MkdirAll(client.Root, os.ModePerm)
	if err != nil && !os.IsExist(err) {
		return err
	}

	subdirs, err := ioutil.ReadDir(client.Root)
	if err != nil {
		return err
	}

	existing := make(map[string]bool)
	needed := make(map[string]bool)

	for _, fileInfo := range subdirs {
		existing[fileInfo.Name()] = true
	}

	for _, resource := range resources {
		needed[resource] = true

		if !existing[resource] {
			err = os.MkdirAll(client.GetResourceDir(resource), os.ModePerm)
			if err != nil {
				return err
			}
		}
	}

	for resource := range existing {
		if !needed[resource] {
			err = os.RemoveAll(client.GetResourceDir(resource))
			if err != nil {
				return err
			}
		}
	}

	return nil
}

func (client *fileClient) GetResourceDir(name string) string {
	return filepath.Join(client.Root, name)
}

func (client *fileClient) GetResourceFilename(resourceName string, resourceVersion uint64) string {
	return filepath.Join(client.GetResourceDir(resourceName), strconv.FormatUint(resourceVersion, 10))
}

func (client *fileClient) IsPresent(resourceName string, resourceVersion uint64) bool {
	_, err := os.Stat(client.GetResourceFilename(resourceName, resourceVersion))
	return err == nil
}

func NewFileClient(config *FileClientConfig) FileClient {
	return &fileClient{config.Root}
}
